package ru.yandex.direct.regions;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import ru.yandex.direct.regions.internal.MetroItem;
import ru.yandex.direct.regions.internal.RegionItem;
import ru.yandex.direct.regions.internal.RegionRoot;

import static com.google.common.base.Preconditions.checkState;
import static ru.yandex.direct.utils.JsonUtils.fromJson;

public class GeoTreeLoader {

    private static final int GEO_FLAG_ENABLED = 1;

    private GeoTreeLoader() {
    }

    /**
     * Фабричный метод
     *
     * @param jsonBody JSON с геоинформацией
     * @return гео-сервис с распарсенным набором регионов
     */
    public static GeoTree build(String jsonBody, GeoTreeType geoTreeType) {
        RegionRoot regionData = fromJson(jsonBody, RegionRoot.class);

        Map<Long, Region> regions = parseRegions(regionData);
        Map<Long, Metro> metro = parseMetro(regionData, regions);

        return new GeoTree(regions, metro, geoTreeType);
    }

    private static Map<Long, Region> parseRegions(RegionRoot regionData) {
        Map<Long, Region> regions = new HashMap<>();

        List<RegionItem> regionItems = regionData.getGeoRegions();
        if (regionItems != null) {
            for (RegionItem item : regionItems) {
                regions.put(item.getId(), convert(item));
            }

            // заполняем родителей, при первом проходе еще нет соответствия id -> region, только region -> id
            for (RegionItem item : regionItems) {
                Region parent = regions.get(item.getParent());
                regions.get(item.getId()).setParent(parent);
            }
        }

        return regions;
    }

    private static Map<Long, Metro> parseMetro(RegionRoot regionData, Map<Long, Region> regions) {
        Map<Long, Metro> metro = new HashMap<>();

        Map<Long, List<MetroItem>> metroStationsData = regionData.getMetroStations();
        if (metroStationsData != null) {
            metroStationsData.forEach((parentId, metroItems) -> {
                Region parent = regions.get(parentId);
                checkState(parent != null, "parent region not found by id = " + parentId);

                metroItems.forEach(item -> metro.put(item.getRegionId(), convert(item, parent)));
            });
        }

        return metro;
    }


    private static Region convert(RegionItem dataItem) {
        boolean geoFlag = dataItem.getGeoFlag() == GEO_FLAG_ENABLED;

        return new Region(dataItem.getId(), dataItem.getType(),
                dataItem.getName(), dataItem.getEname(), dataItem.getUaName(), dataItem.getTrName(), geoFlag
        );
    }

    private static Metro convert(MetroItem metroItem, Region parent) {
        return new Metro(metroItem.getRegionId(), parent, metroItem.getName());
    }
}
