package ru.yandex.direct.regions;

import java.util.Objects;

/**
 * Содержит информацию об одном регионе из геоинформации
 */
public class Region {
    /**
     * Континент
     */
    public static final int REGION_TYPE_CONTINENT = 1;

    /**
     * Страна
     */
    public static final int REGION_TYPE_COUNTRY = 3;

    /**
     * Округ
     */
    public static final int REGION_TYPE_DISTRICT = 4;

    /**
     * Область/провинция
     */
    public static final int REGION_TYPE_PROVINCE = 5;

    /**
     * Город
     */
    public static final int REGION_TYPE_TOWN = 6;

    /**
     * Заморская территория
     */
    public static final int REGION_TYPE_OVERSEAS_TERRITORY = 12;

    /**
     * Удаленный регион
     */
    public static final long DELETED_REGION_ID = -1L;

    /**
     * Весь мир
     */
    public static final long GLOBAL_REGION_ID = 0L;

    /**
     * Содружество независимых государств
     */
    public static final long SNG_REGION_ID = 166L;

    /**
     * Европа
     */
    public static final long EUROPE_REGION_ID = 111L;

    /**
     * Идентификатор региона для России
     */
    public static final long RUSSIA_REGION_ID = 225L;

    /**
     * Идентификатор региона для Белоруссии
     */
    public static final long BY_REGION_ID = 149L;

    /**
     * Идентификатор региона для Узбекстана
     */
    public static final long UZBEKISTAN_REGION_ID = 171L;

    /**
     * Идентификатор региона для Украины
     */
    public static final long UKRAINE_REGION_ID = 187L;

    /**
     * Идентификатор региона для Крыма
     */
    public static final long CRIMEA_REGION_ID = 977L;

    /**
     * Идентификатор региона в Симферополе (Крым)
     */
    public static final long SIMFEROPOL_REGION_ID = 146L;

    /**
     * Идентификатор региона для Казахстана
     */
    public static final long KAZAKHSTAN_REGION_ID = 159L;

    /**
     * Идентификатор региона для Турции
     */
    public static final long TURKEY_REGION_ID = 983L;

    /**
     * Идентификатор региона для г.Москвы и Московской области
     */
    public static final long MOSCOW_AND_MOSCOW_PROVINCE_REGION_ID = 1L;

    /**
     * Идентификатор региона для г.Москвы
     */
    public static final long MOSCOW_REGION_ID = 213L;

    /**
     * Идентификаторы остальных регионов.
     * Для однообразия наименования констант соответствует полю ename из файла regions.json
     */
    public static final long KYIV_OBLAST_REGION_ID = 20544L;
    public static final long KYIV_REGION_ID = 143L;
    public static final long ODESA_DISTRICT_REGION_ID = 20541L;
    public static final long SAINT_PETERSBURG_AND_LENINGRAD_OBLAST_REGION_ID = 10174L;
    public static final long SAINT_PETERSBURG_REGION_ID = 2L;
    public static final long SVERDLOVSK_OBLAST_REGION_ID = 11162L;
    public static final long NOVOSIBIRSK_OBLAST_REGION_ID = 11316L;
    public static final long NIZHNY_NOVGOROD_OBLAST_REGION_ID = 11079L;
    public static final long YAROSLAVL_OBLAST_REGION_ID = 10841L;
    public static final long CENTRAL_FEDERAL_DISTRICT_REGION_ID = 3L;
    public static final long BELGOROD_OBLAST_REGION_ID = 10645L;
    public static final long VORONEZH_OBLAST_REGION_ID = 10672L;
    public static final long SOUTH_FEDERAL_DISTRICT_REGION_ID = 26L;
    public static final long NORTHWESTERN_FEDERAL_DISTRICT_REGION_ID = 17L;
    public static final long NORTH_CAUCASIAN_FEDERAL_DISTRICT = 102444L;
    public static final long REPUBLIC_OF_TATARSTAN_REGION_ID = 11119L;
    public static final long URAL_FEDERAL_DISTRICT_REGION_ID = 52L;
    public static final long SIBERIAN_FEDERAL_DISTRICT_REGION_ID = 59L;
    public static final long FAR_EASTERN_FEDERAL_DISTRICT_REGION_ID = 73L;
    public static final long VOLGA_FEDERAL_DISTRICT_REGION_ID = 40L;
    public static final long REPUBLIC_OF_BASHKORTOSTAN_REGION_ID = 11111L;
    public static final long PERM_KRAI_REGION_ID = 11108L;
    public static final long ULYANOVSK_OBLAST_REGION_ID = 11153L;
    public static final long SAMARA_OBLAST_REGION_ID = 11131L;
    public static final long KIROV_OBLAST_REGION_ID = 11070L;
    public static final long UDMURT_REPUBLIC_REGION_ID = 11148L;
    public static final long MARI_EL_REPUBLIC_REGION_ID = 11077L;
    public static final long CHUVASH_REPUBLIC_REGION_ID = 11156L;
    public static final long REPUBLIC_OF_MORDOVIA_REGION_ID = 11117L;
    public static final long ISTANBUL_REGION_ID = 103728L;
    public static final long ASIA_REGION_ID = 183L;
    public static final long AFRICA_REGION_ID = 241L;
    public static final long NORTH_AMERICA_REGION_ID = 10002L;
    public static final long SOUTH_AMERICA_REGION_ID = 10003L;
    public static final long AUSTRALIA_AND_OCEANIA_REGION_ID = 138L;

    private long id;
    /**
     * Ссылка на родительский регион. Для корневого региона ссылается сам на себя
     */
    private Region parent;
    /**
     * См. константы REGION_TYPE_XXX
     */
    private int type;
    private boolean geoFlag;
    private String nameRu;
    private String nameEn;
    private String nameUa;
    private String nameTr;

    /**
     * Создает экземпляр региона из исходных данных
     *
     * @param id      идентификатор региона
     * @param type    тип региона (из геоинформации)
     * @param nameRu  название региона по-русски
     * @param nameEn  название региона по-английски
     * @param nameUa  название региона по-украински
     * @param nameTr  название региона по-турецки
     * @param geoFlag геофлаг
     */
    Region(long id, int type, String nameRu, String nameEn, String nameUa, String nameTr, boolean geoFlag) {
        this.id = id;
        this.type = type;
        this.nameRu = nameRu;
        this.nameEn = nameEn;
        this.nameUa = nameUa;
        this.nameTr = nameTr;
        this.geoFlag = geoFlag;

    }

    /**
     * @return идентификатор региона
     */
    public long getId() {
        return id;
    }

    /**
     * @return регион, содержащий данный регион
     */
    public Region getParent() {
        return parent;
    }

    /**
     * @param parent регион, содержащий данный регион
     */
    void setParent(Region parent) {
        this.parent = parent;
    }

    /**
     * @return тип региона
     */
    public int getType() {
        return type;
    }

    /**
     * @return название типа региона
     */
    public String getTypeName() {
        switch (type) {
            case 0:
                return "World";
            case 1:
                return "Continent";
            case 2:
                return "Region";
            case 3:
                return "Country";
            case 4:
                return "Administrative area";
            case 5:
                return "Administrative area";
            case 6:
                return "City";
            case 7:
                return "Village";
            case 8:
                return "City district";
            case 10:
                return "District";
            default:
                return "";
        }
    }

    /**
     * @return название региона по-русски
     */
    public String getNameRu() {
        return nameRu;
    }

    /**
     * @return название региона по-английски
     */
    public String getNameEn() {
        return nameEn;
    }

    /**
     * @return название региона по-украински
     */
    public String getNameUa() {
        return nameUa;
    }

    /**
     * @return название региона по-турецки
     */
    public String getNameTr() {
        return nameTr;
    }

    /**
     * Проверяет, что название города совпадает с одним из названий на разных языках
     */
    public boolean equalsAnyNameIgnoreCase(String name) {
        Objects.requireNonNull(name, "name");

        String trimmedName = name.trim();

        return nameRu.equalsIgnoreCase(trimmedName)
                || nameEn.equalsIgnoreCase(trimmedName)
                || nameUa.equalsIgnoreCase(trimmedName)
                || nameTr.equalsIgnoreCase(trimmedName);
    }

    /**
     * @return геофлаг (Нужно ли отображать название данного региона в объявлении)
     */
    public boolean getGeoFlag() {
        return geoFlag;
    }
}
