package ru.yandex.direct.regions;

import java.util.function.Supplier;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SwitchingGeoTreeFactory implements GeoTreeFactory {
    private static final Logger logger = LoggerFactory.getLogger(SwitchingGeoTreeFactory.class);

    private final Supplier<Boolean> switcher;
    private final GeoTreeFactory prev;
    private final GeoTreeFactory next;

    public SwitchingGeoTreeFactory(Supplier<Boolean> switcher, GeoTreeFactory prev, GeoTreeFactory next) {
        this.switcher = new FailSafeBooleanSupplier(switcher, false);
        this.prev = prev;
        this.next = next;
    }

    private GeoTreeFactory current() {
        if (switcher.get()) {
            return next;
        } else {
            return prev;
        }
    }

    /**
     * Тесты подсветили, что при недоступности БД {@code switcher} может кинуть Exception.
     * Для надёжности оборачиваем в try-catch
     */
    private static class FailSafeBooleanSupplier implements Supplier<Boolean> {
        private final Supplier<Boolean> origin;
        private final boolean defaultValue;

        private FailSafeBooleanSupplier(Supplier<Boolean> origin, boolean defaultValue) {
            this.origin = origin;
            this.defaultValue = defaultValue;
        }

        @Override
        public Boolean get() {
            try {
                return origin.get();
            } catch (RuntimeException e) {
                logger.error("Can't get switch-flag. Use {} as default", defaultValue, e);
                return defaultValue;
            }
        }
    }

    @Override
    public GeoTree getApiGeoTree() {
        return current().getApiGeoTree();
    }

    @Override
    public GeoTree getRussianGeoTree() {
        return current().getRussianGeoTree();
    }

    @Override
    public GeoTree getGlobalGeoTree() {
        return current().getGlobalGeoTree();
    }

    @Override
    public GeoTree getTranslocalGeoTree(long clientCountryId) {
        return current().getTranslocalGeoTree(clientCountryId);
    }
}
