package ru.yandex.direct.result;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.DefectInfo;
import ru.yandex.direct.validation.result.PathNode;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Preconditions.checkArgument;
import static java.util.Collections.emptyList;
import static java.util.Collections.emptySet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.result.PathHelper.index;
import static ru.yandex.direct.validation.result.PathHelper.path;

public class MassResult<R> extends Result<List<Result<R>>> {

    private OperationMeta operationMeta;
    private int successfulCount;
    private int errorCount;

    public MassResult(List<Result<R>> results,
                      ValidationResult<?, Defect> validationResult, ResultState state) {
        super(results, validationResult, state);
        checkArgument(state != ResultState.CANCELED, "MassResult can not be in canceled state");
        if (results != null) {
            // Считаем, что после создания не меняются ни elementsResults, ни massValidation, и count'ы можно запомнить
            this.successfulCount = (int) results.stream().filter(Result::isSuccessful).count();
            this.errorCount = results.size() - successfulCount;
        }
    }

    public int getSuccessfulCount() {
        return successfulCount;
    }

    public int getErrorCount() {
        return errorCount;
    }

    public Optional<OperationMeta> getOperationMeta() {
        return Optional.ofNullable(operationMeta);
    }

    public MassResult<R> withOperationMeta(OperationMeta operationMeta) {
        this.operationMeta = operationMeta;
        return this;
    }

    public static <T> MassResult<T> emptyMassAction() {
        return new MassResult<>(emptyList(), ValidationResult.success(emptyList()), ResultState.SUCCESSFUL);
    }

    public static <T, V> MassResult<T> successfulMassAction(List<? extends T> resultItems,
                                                            ValidationResult<? extends List<? extends V>, Defect> massValidation,
                                                            Set<Integer> canceledElementIndexes) {
        checkArgument(!massValidation.hasErrors(),
                "successful mass action result must not contain operational errors");
        checkArgument(resultItems.size() == massValidation.getValue().size(),
                "Size of result items should match size of value in validation result");

        List<Result<T>> elementsResults = createElementsResults(resultItems, massValidation, canceledElementIndexes);
        return new MassResult<>(elementsResults, massValidation, ResultState.SUCCESSFUL);
    }

    public static <T, V> MassResult<T> successfulMassAction(List<? extends T> resultItems,
                                                            ValidationResult<? extends List<? extends V>, Defect> massValidation) {
        return successfulMassAction(resultItems, massValidation, emptySet());
    }

    public static <T, V> MassResult<T> brokenMassAction(List<? extends T> resultItems,
                                                        ValidationResult<? extends List<? extends V>, Defect> massValidation) {
        if (massValidation.hasErrors()) {
            return new MassResult<>(null,
                    massValidation,
                    ResultState.BROKEN);
        } else {
            List<Result<T>> elementsResults = createElementsResults(resultItems, massValidation, emptySet());
            return new MassResult<>(elementsResults, massValidation, ResultState.SUCCESSFUL);
        }
    }

    private static <T, V> List<Result<T>> createElementsResults(List<? extends T> resultItems,
                                                                ValidationResult<? extends List<? extends V>, Defect> massValidation,
                                                                Set<Integer> canceledElementIndexes) {
        Map<PathNode, ValidationResult<?, Defect>> validations = massValidation.getSubResults();
        List<Result<T>> results = new ArrayList<>(resultItems.size());
        for (int i = 0; i < resultItems.size(); i++) {
            ValidationResult<?, Defect> validation = validations.get(index(i));
            boolean hasErrors = validation != null && validation.hasAnyErrors();
            boolean canceled = canceledElementIndexes.contains(i);
            checkArgument(!(canceled && hasErrors), "canceled item %s must not contain errors", i);

            if (hasErrors) {
                results.add(broken(validation));
            } else if (canceled) {
                results.add(canceled(validation));
            } else {
                T resultItem = resultItems.get(i);
                results.add(successful(resultItem, validation));
            }
        }

        return results;
    }

    /**
     * @return только ошибки уровня операции.
     */
    @Override
    public List<DefectInfo<Defect>> getErrors() {
        if (getValidationResult() == null) {
            return emptyList();
        }
        return mapList(getValidationResult().getErrors(),
                defectDefinition -> new DefectInfo<>(path(), getValidationResult().getValue(), defectDefinition));
    }

    /**
     * @return только предупреждения уровня операции.
     */
    @Override
    public List<DefectInfo<Defect>> getWarnings() {
        if (getValidationResult() == null) {
            return emptyList();
        }
        return mapList(getValidationResult().getWarnings(),
                defectDefinition -> new DefectInfo<>(path(), getValidationResult().getValue(), defectDefinition));
    }

    /**
     * Метод явно задаёт количества успешно обработанных объектов, и обработанных с ошибкой.
     * Эта информация используется для вычисления количества баллов, потраченных на выполнение операции.
     *
     * @param successfulCount количество успешно обработанных объектов
     * @param errorCount      количество объектов, при обработке которых возникли ошибки
     */
    public MassResult<R> withCounts(int successfulCount, int errorCount) {
        this.successfulCount = successfulCount;
        this.errorCount = errorCount;
        return this;
    }

    @Override
    public int getSuccessfulObjectsCount() {
        return successfulCount;
    }

    @Override
    public int getUnsuccessfulObjectsCount() {
        return errorCount;
    }

    public Result<R> get(int i) {
        return getResult().get(i);
    }

    public List<Result<R>> toResultList() {
        return getResult();
    }
}
