package ru.yandex.direct.result;

import java.util.Collection;
import java.util.List;

import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.DefectInfo;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Preconditions.checkArgument;
import static java.util.Collections.emptyList;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;

/**
 * Результат пишущей операции.
 * <p>
 * При успешном выполнении операции результат имеет статус {@link ResultState#SUCCESSFUL}
 * и содержит результат (id или более расширенную информацию о выполнении).
 * В случае, если исходные данные не прошли валидацию, результат должен иметь статус
 * {@link ResultState#BROKEN}, не содержать значение и должен содержать ошибки валидации
 * в {@link #getErrors()}. В случае, если выполнение операции было отменено результат должен иметь
 * статус {@link ResultState#CANCELED} и не содержать ни значений, ни ошибок.
 *
 * @param <R> тип значения в результате
 */
public class Result<R> {
    private final ResultState state;
    private final R result;
    private final ValidationResult<?, Defect> validationResult;

    public Result(R result,
                  ValidationResult<?, Defect> validationResult,
                  ResultState state) {
        this.result = result;
        this.validationResult = validationResult;
        this.state = state;
    }

    public Result(Result<R> another) {
        this.result = another.result;
        this.validationResult = ifNotNull(another.validationResult, ValidationResult::new);
        this.state = another.state;
    }

    public static <R> Result<R> successful(R result) {
        return successful(result, null);
    }

    public static <R> Result<R> successful(R result, ValidationResult<?, Defect> validationResult) {
        checkArgument(validationResult == null || !validationResult.hasAnyErrors(),
                "Successful result must not contain errors");
        return new Result<>(result, validationResult, ResultState.SUCCESSFUL);
    }

    public static <R> Result<R> broken(ValidationResult<?, Defect> validationResult) {
        checkArgument(validationResult != null && validationResult.hasAnyErrors(),
                "Broken result must contain errors");
        return new Result<>(null, validationResult, ResultState.BROKEN);
    }

    public static <R> Result<R> canceled(ValidationResult<?, Defect> validationResult) {
        checkArgument(validationResult == null || !validationResult.hasAnyErrors(),
                "Canceled result must not contain errors");
        return new Result<>(null, validationResult, ResultState.CANCELED);
    }

    public int getSuccessfulObjectsCount() {
        if (result instanceof Collection) {
            // result - результат простой операции (как get), где "все элементы успешные"
            return ((Collection) result).size();
        }

        return isSuccessful() ? 1 : 0;
    }

    public int getUnsuccessfulObjectsCount() {
        return 0;
    }

    /**
     * @return Объект результата при успешном выполнении, в противном случае {@code null}
     */
    public R getResult() {
        return result;
    }

    /**
     * Возвращает все ошибки
     *
     * @return список ошибок
     */
    public List<DefectInfo<Defect>> getErrors() {
        return validationResult != null ? validationResult.flattenErrors() : emptyList();
    }

    /**
     * Возвращает все предупреждения
     *
     * @return список предупреждений
     */
    public List<DefectInfo<Defect>> getWarnings() {
        return validationResult != null ? validationResult.flattenWarnings() : emptyList();
    }

    public ValidationResult<?, Defect> getValidationResult() {
        return validationResult;
    }

    /**
     * Возвращает состояние результата
     */
    public ResultState getState() {
        return state;
    }

    /**
     * @return {@code true}, если результат положительный и содержит объект результата
     * {@code false}, если результат отрицательный.
     */
    public boolean isSuccessful() {
        return state == ResultState.SUCCESSFUL;
    }
}
