package ru.yandex.direct.rotor.client;

import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.asynchttp.ErrorResponseWrapperException;
import ru.yandex.direct.asynchttp.ParallelFetcherFactory;
import ru.yandex.direct.asynchttp.Result;
import ru.yandex.direct.asynchttp.ResultUtils;
import ru.yandex.direct.http.smart.annotations.Json;
import ru.yandex.direct.http.smart.annotations.ResponseHandler;
import ru.yandex.direct.http.smart.converter.ResponseConverterFactory;
import ru.yandex.direct.http.smart.core.Call;
import ru.yandex.direct.http.smart.core.Smart;
import ru.yandex.direct.http.smart.http.Body;
import ru.yandex.direct.http.smart.http.Header;
import ru.yandex.direct.http.smart.http.Headers;
import ru.yandex.direct.http.smart.http.POST;
import ru.yandex.direct.rotor.client.model.RotorImageResponse;
import ru.yandex.direct.rotor.client.model.RotorImageResponseConverter;
import ru.yandex.direct.rotor.client.model.RotorPayload;
import ru.yandex.direct.rotor.client.model.RotorResponse;
import ru.yandex.direct.rotor.client.model.RotorResponseConverter;
import ru.yandex.direct.tvm.TvmIntegration;
import ru.yandex.direct.tvm.TvmService;

@ParametersAreNonnullByDefault
public class RotorClient {
    private static final Logger logger = LoggerFactory.getLogger(RotorClient.class);

    private final String rotorSource;
    private final Api api;
    private final TvmIntegration tvmIntegration;
    private final TvmService tvmService;
    @Nullable
    private final String clientId;

    public RotorClient(String rotorSource, String rotorUrl, ParallelFetcherFactory parallelFetcherFactory,
                       TvmIntegration tvmIntegration, TvmService tvmService, @Nullable String clientId) {
        this.rotorSource = rotorSource;
        this.tvmIntegration = tvmIntegration;
        this.tvmService = tvmService;
        this.clientId = clientId;
        this.api = createApi(rotorUrl, parallelFetcherFactory);
    }

    private Api createApi(String url, ParallelFetcherFactory parallelFetcherFactory) {
        return Smart.builder()
                .withParallelFetcherFactory(parallelFetcherFactory)
                .withResponseConverterFactory(ResponseConverterFactory.builder()
                        .addConverters(
                                new RotorImageResponseConverter(),
                                new RotorResponseConverter()
                        )
                        .build())
                .withProfileName("rotor_service")
                .withBaseUrl(url)
                .build()
                .create(Api.class);
    }

    public interface Api {
        @POST("/v1/rotor/execute/")
        @Headers("Content-Type: application/json")
        @ResponseHandler(parserClass = RotorResponseConverter.class)
        Call<RotorResponse> get(@Body @Json RotorPayload rotorRequest);

        @POST("/v1/rotor/execute/")
        @Headers("Content-Type: application/json")
        @ResponseHandler(parserClass = RotorResponseConverter.class)
        Call<RotorResponse> get(@Body @Json RotorPayload rotorRequest,
                                @Header("X-Ya-Client-Id") String clientId);

        @POST("/v1/rotor/execute/png/")
        @Headers("Content-Type: application/json")
        @ResponseHandler(parserClass = RotorImageResponseConverter.class)
        Call<RotorImageResponse> getImage(@Body @Json RotorPayload rotorRequest);
    }

    public RotorImageResponse getScreenshotFromUrl(String url, long width, long height) {
        RotorPayload rotorRequest = new RotorPayload();
        rotorRequest.setTvmServiceTicket(tvmIntegration.getTicket(tvmService));
        rotorRequest.setSource(rotorSource);
        rotorRequest.setUrl(url);
        rotorRequest.getOptions().getViewPortSize().setWidth(width);
        rotorRequest.getOptions().getViewPortSize().setHeight(height);

        return executeImageRequest(rotorRequest);
    }

    public RotorImageResponse getScreenshotFromHtml(String html, long width, long height) {
        RotorPayload rotorRequest = new RotorPayload();
        rotorRequest.setTvmServiceTicket(tvmIntegration.getTicket(tvmService));
        rotorRequest.setSource(rotorSource);
        // fake but mandatory, see https://st.yandex-team.ru/BANNERSTORAGE-5026#1491921934000)
        // причём в конце обязательно должен быть слеш (для нового GoRotor)
        // см https://st.yandex-team.ru/DIRECT-153102#614ca2cfa18849001a2477b1
        rotorRequest.setUrl("https://storage.mds.yandex.net/");
        rotorRequest.getOptions().getViewPortSize().setWidth(width);
        rotorRequest.getOptions().getViewPortSize().setHeight(height);
        rotorRequest.setHttpResponse(String.join("\n", "HTTP/1.1 200 Ok", "Cache-Control: no-cache",
                "Content-Type: text/html; charset=utf-8", "\n", html));

        return executeImageRequest(rotorRequest);
    }

    private RotorImageResponse executeImageRequest(RotorPayload rotorRequest) {
        logger.info("executing request to Rotor \"{}\"", api.getImage(rotorRequest).getRequest());

        Result<RotorImageResponse> result = api.getImage(rotorRequest).execute();

        if (result.getSuccess() == null && result.getErrors() != null) {
            logger.error("can't get screenshot from Rotor ({})", result.getErrors()
                    .stream()
                    .map(error -> error.toString() +
                            (error instanceof ErrorResponseWrapperException &&
                                    ((ErrorResponseWrapperException) error).getResponse() != null ?
                                    " (" + ((ErrorResponseWrapperException) error).getResponse().toString() + ")" : ""))
                    .collect(Collectors.toList()));
        }
        return result.getSuccess();
    }

    public RotorResponse get(String url) {
        RotorPayload rotorRequest = new RotorPayload();
        rotorRequest.setTvmServiceTicket(tvmIntegration.getTicket(tvmService));
        rotorRequest.setSource(rotorSource);
        rotorRequest.setUrl(url);
        rotorRequest.getOptions().getOutputFormat().setHtml(true);
        rotorRequest.getOptions().getOutputFormat().setPng(false);
        rotorRequest.getOptions().setTrace(true);

        logger.info("executing request to Rotor \"{}\"", api.get(rotorRequest).getRequest());

        var call = clientId != null
                ? api.get(rotorRequest, clientId)
                : api.get(rotorRequest);
        var result = call.execute();
        if (result.getSuccess() == null && result.getErrors() != null) {
            var errorMessage = ResultUtils.getErrorMessage(result);
            logger.error("can't get url from Rotor {}", errorMessage);
        }
        return result.getSuccess();
    }
}
