package ru.yandex.direct.scheduler;

import java.lang.annotation.ElementType;
import java.lang.annotation.Inherited;
import java.lang.annotation.Repeatable;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;

import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;

import ru.yandex.direct.utils.Condition;

import static org.springframework.beans.factory.config.ConfigurableBeanFactory.SCOPE_PROTOTYPE;

/**
 * Расписание для процесса, запускамого шедулероe
 */
@Target(ElementType.TYPE)
@Retention(RetentionPolicy.RUNTIME)
@Repeatable(Hourglasses.class)
@Component
@Inherited
@Scope(SCOPE_PROTOTYPE)
public @interface Hourglass {
    int PERIOD_NOT_SPECIFIED = -1;
    String CRON_EXPRESSION_NOT_SPECIFIED = "";

    /**
     * Раз в сколько секунд запускать джоб - взаимоисключающее поле с {@link #cronExpression()}, имеет приоритет над ним
     *
     * @return Промежуток времени в секундах
     */
    int periodInSeconds() default PERIOD_NOT_SPECIFIED;

    /**
     * Cron-выражение, определяющее, когда должен запускатться джоб - взаимоисключающе поле с
     * {@link #periodInSeconds()},
     * при указании обоих полей - не используется
     * <p>
     * Примеры:
     * 0 20 4 * * ? - запускаться в любой день любого месяца в 04:20:00, ? означает "не учитывать день недели"
     *
     * @return Cron-выражение
     * @see
     * <a href="http://man7.org/linux/man-pages/man5/crontab.5.html">CronTrigger Format</a>
     */
    String cronExpression() default CRON_EXPRESSION_NOT_SPECIFIED;

    /**
     * Класс (реализующий {@link Condition}), содержащий логику "нужно ли использовать это расписание".
     * Должен быть помечен аннотацией {@link Component}, чтобы быть доступным в виде бина в текущем контексте.
     * <p>
     * Шедулер запланирует (использует) данное расписание только при истинном значении {@link Condition#evaluate()}}.
     *
     * @return класс с логикой если был задан, или {@link AlwaysTrue}
     */
    Class<? extends Condition> needSchedule() default AlwaysTrue.class;
}
