package ru.yandex.direct.scheduler.hourglass.implementations;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.TimeUnit;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeansException;
import org.springframework.context.ApplicationContext;

import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.HourglassDaemon;
import ru.yandex.direct.scheduler.HourglassStretchPeriod;
import ru.yandex.direct.scheduler.HourglassStretchingDisabled;
import ru.yandex.direct.scheduler.hourglass.HourglassJob;
import ru.yandex.direct.scheduler.hourglass.ParamDescription;
import ru.yandex.direct.scheduler.hourglass.TaskDescription;
import ru.yandex.direct.scheduler.hourglass.implementations.schedule.ScheduleInfoImpl;
import ru.yandex.direct.scheduler.hourglass.implementations.schedule.modifiers.ModifierDataWithTypeImpl;
import ru.yandex.direct.scheduler.hourglass.implementations.schedule.modifiers.RandomDeltaCalculator;
import ru.yandex.direct.scheduler.hourglass.implementations.schedule.modifiers.RandomStartTimeData;
import ru.yandex.direct.scheduler.hourglass.implementations.schedule.strategies.ScheduleDataWithTypeImpl;
import ru.yandex.direct.scheduler.hourglass.implementations.schedule.strategies.data.ScheduleCronData;
import ru.yandex.direct.scheduler.hourglass.implementations.schedule.strategies.data.ScheduleDaemonData;
import ru.yandex.direct.scheduler.hourglass.implementations.schedule.strategies.data.SchedulePeriodicData;
import ru.yandex.direct.scheduler.hourglass.schedule.ScheduleInfo;
import ru.yandex.direct.scheduler.hourglass.schedule.modifiers.ModifierType;
import ru.yandex.direct.scheduler.hourglass.schedule.strategies.ScheduleType;

import static java.util.stream.Collectors.toUnmodifiableList;
import static ru.yandex.direct.scheduler.Hourglass.CRON_EXPRESSION_NOT_SPECIFIED;
import static ru.yandex.direct.scheduler.Hourglass.PERIOD_NOT_SPECIFIED;

public class JobScheduleInfoFactory {
    private static final Logger logger = LoggerFactory.getLogger(HourglassScheduler.class);
    private final ApplicationContext context;
    private final RandomDeltaCalculator randomDeltaCalculator = new RandomDeltaCalculator();

    public JobScheduleInfoFactory(ApplicationContext context) {
        this.context = context;
    }

    public ScheduleInfo getScheduleInfo(TaskDescription taskDescription, ParamDescription paramDescription) {
        var jobClass = taskDescription.getTaskClass();
        List<Hourglass> hourglasses = getScheduleAnnotations(jobClass);
        List<ScheduleDataWithTypeImpl> nextRunCalcStrategies = getNextRunCalcStrategies(jobClass, hourglasses);
        List<ModifierDataWithTypeImpl> scheduleModifierParams = getScheduleModifiers(taskDescription, paramDescription,
                hourglasses);

        return new ScheduleInfoImpl(nextRunCalcStrategies, scheduleModifierParams);
    }

    private List<Hourglass> getScheduleAnnotations(Class<? extends HourglassJob> jobClass) {
        List<Hourglass> schedules = new ArrayList<>();

        for (var schedule : jobClass.getAnnotationsByType(Hourglass.class)) {
            try {
                if (context.getBean(schedule.needSchedule()).evaluate()) {
                    schedules.add(schedule);
                } else {
                    logger.debug("Skip schedule({}) for Job {} due to needSchedule condition",
                            getScheduleString(schedule), jobClass.getCanonicalName());
                }
            } catch (BeansException ex) {
                logger.error(String.format("Exception occurred while checking schedule (%s) for Job %s: ",
                        getScheduleString(schedule), jobClass.getCanonicalName()), ex);
            }
        }

        return schedules;
    }

    private String getScheduleString(Hourglass schedule) {
        return String.format("period = %s, cron = %s, condition class = %s", schedule.periodInSeconds(),
                schedule.cronExpression(), schedule.needSchedule().getName());
    }

    private List<ScheduleDataWithTypeImpl> getNextRunCalcStrategies(Class<? extends HourglassJob> jobClass,
                                                                    List<Hourglass> schedules) {
        var suitableSchedules = schedules.stream()
                .map(suitableSchedule -> getJobScheduleDataWithType(jobClass, suitableSchedule))
                .filter(Objects::nonNull)
                .collect(toUnmodifiableList());

        if (suitableSchedules.isEmpty()) {
            logger.info("Adding job {} with far-future strategy", jobClass.getCanonicalName());

            suitableSchedules = List.of(
                    new ScheduleDataWithTypeImpl(ScheduleType.FAR_FUTURE, null));
        }

        return suitableSchedules;
    }

    private List<ModifierDataWithTypeImpl> getScheduleModifiers(TaskDescription taskDescription,
                                                                ParamDescription paramDescription,
                                                                List<Hourglass> schedules) {

        if (taskDescription.getTaskClass().isAnnotationPresent(HourglassStretchingDisabled.class)) {
            return List.of();
        }

        HourglassStretchPeriod[] hourglassStretchPeriod =
                taskDescription.getTaskClass().getAnnotationsByType(HourglassStretchPeriod.class);
        Integer stretchPeriod = null;
        if (hourglassStretchPeriod.length > 0) {
            stretchPeriod = hourglassStretchPeriod[0].value();
        } else {
            for (Hourglass hourglass : schedules) {
                if (hourglass.periodInSeconds() != PERIOD_NOT_SPECIFIED) {
                    stretchPeriod = hourglass.periodInSeconds();
                    break;
                }
            }
        }
        if (Objects.isNull(stretchPeriod)) {
            return List.of();
        }
        var randomData = randomDeltaCalculator.calculateDelta(stretchPeriod, taskDescription, paramDescription);
        return List.of(new ModifierDataWithTypeImpl(ModifierType.RANDOM_START_TIME,
                new RandomStartTimeData(randomData)));
    }


    private ScheduleDataWithTypeImpl getJobScheduleDataWithType(Class<? extends HourglassJob> jobClass,
                                                                Hourglass schedule) {
        ScheduleDataWithTypeImpl scheduleDataWithType;

        if (jobClass.isAnnotationPresent(HourglassDaemon.class)) {
            scheduleDataWithType = new ScheduleDataWithTypeImpl(ScheduleType.DAEMON,
                    new ScheduleDaemonData(schedule.periodInSeconds()));

        } else if (!schedule.cronExpression().equals(CRON_EXPRESSION_NOT_SPECIFIED)) {
            scheduleDataWithType = new ScheduleDataWithTypeImpl(ScheduleType.CRON,
                    new ScheduleCronData(schedule.cronExpression()));

        } else if (schedule.periodInSeconds() != PERIOD_NOT_SPECIFIED) {
            scheduleDataWithType = new ScheduleDataWithTypeImpl(ScheduleType.PERIODIC,
                    new SchedulePeriodicData(schedule.periodInSeconds(), TimeUnit.SECONDS));

        } else {
            logger.error("No schedule defined in Hourglass annotation for the following JobKey: {}",
                    jobClass.getCanonicalName());
            scheduleDataWithType = null;
        }

        return scheduleDataWithType;
    }
}
