package ru.yandex.direct.scheduler.hourglass.implementations;

import java.time.Instant;
import java.util.function.BinaryOperator;

import ru.yandex.direct.hourglass.TaskProcessingResult;
import ru.yandex.direct.scheduler.hourglass.schedule.NextRunCalculator;
import ru.yandex.direct.scheduler.hourglass.schedule.ScheduleInfo;
import ru.yandex.direct.scheduler.hourglass.schedule.modifiers.ModifierDataWithType;
import ru.yandex.direct.scheduler.hourglass.schedule.modifiers.NextRunModifierFactory;
import ru.yandex.direct.scheduler.hourglass.schedule.strategies.NextRunStrategiesFactory;
import ru.yandex.direct.scheduler.hourglass.schedule.strategies.ScheduleDataWithType;

public class ScheduleInfoProcessor {

    private final NextRunStrategiesFactory nextRunStrategiesFactory;
    private final NextRunModifierFactory nextRunModifierFactory;

    public ScheduleInfoProcessor(NextRunStrategiesFactory nextRunStrategiesFactory,
                                 NextRunModifierFactory nextRunModifierFactory) {
        this.nextRunStrategiesFactory = nextRunStrategiesFactory;
        this.nextRunModifierFactory = nextRunModifierFactory;
    }

    public NextRunCalculator getNextRunCalculator(ScheduleInfo scheduleInfo, BinaryOperator<Instant> reducer) {
        return new NextRunCalculator(taskProcessingResult -> getNextRunTime(scheduleInfo, reducer,
                taskProcessingResult));
    }

    public Instant getNextRunTime(ScheduleInfo scheduleInfo, BinaryOperator<Instant> reducer,
                                  TaskProcessingResult taskProcessingResult) {

        Instant nextRun =
                scheduleInfo.getStrategies()
                        .stream()
                        .map(strategy -> getNextDateFromStrategy(strategy, taskProcessingResult))
                        .reduce(reducer)
                        .orElse(Instant.now());

        for (var modifier : scheduleInfo.getModifiers()) {
            nextRun = getNextDateFromModifier(modifier, nextRun);
        }

        return nextRun;
    }

    private Instant getNextDateFromStrategy(ScheduleDataWithType scheduleDataWithType,
                                            TaskProcessingResult taskProcessingResult) {
        var strategy = nextRunStrategiesFactory.getNextRunStrategy(scheduleDataWithType.getType());
        return strategy.getNextDate(taskProcessingResult, scheduleDataWithType.getData());
    }

    private Instant getNextDateFromModifier(ModifierDataWithType modifierDataWithType, Instant prevNextRun) {
        var modifier = nextRunModifierFactory.getModifier(modifierDataWithType.getType());
        return modifier.modify(prevNextRun, modifierDataWithType.getData());
    }

}
