package ru.yandex.direct.scheduler.hourglass.implementations;

import java.util.Map;

import javax.annotation.PostConstruct;

import org.springframework.context.ApplicationContext;

import ru.yandex.direct.hourglass.TaskHooks;
import ru.yandex.direct.hourglass.storage.Job;
import ru.yandex.direct.scheduler.HourglassDaemon;
import ru.yandex.direct.scheduler.JobInterceptorsList;
import ru.yandex.direct.scheduler.hourglass.HourglassJob;
import ru.yandex.direct.scheduler.hourglass.TaskDescription;
import ru.yandex.direct.scheduler.hourglass.TaskFactory;
import ru.yandex.direct.scheduler.hourglass.TaskListProvider;
import ru.yandex.direct.scheduler.hourglass.schedule.ScheduleInfoConverter;
import ru.yandex.direct.scheduler.support.BaseDirectJob;
import ru.yandex.direct.scheduler.support.DaemonJobWrapper;
import ru.yandex.direct.scheduler.support.JobExecutingMonitor;
import ru.yandex.direct.scheduler.support.PeriodicJobWrapper;

import static java.util.stream.Collectors.toMap;

public class TaskFactoryImpl implements TaskFactory {
    private final ApplicationContext context;
    private final TaskListProvider taskListProvider;
    private final JobInterceptorsList interceptorsList;
    private final JobExecutingMonitor jobExecutingMonitor;
    private Map<String, TaskDescription> taskIdTaskDescription;
    private final ScheduleInfoConverter scheduleInfoConverter;
    private final ScheduleInfoProcessor scheduleInfoProcessor;

    public TaskFactoryImpl(ApplicationContext context, TaskListProvider taskListProvider,
                           JobInterceptorsList interceptorsList,
                           JobExecutingMonitor jobExecutingMonitor,
                           ScheduleInfoConverter scheduleInfoConverter,
                           ScheduleInfoProcessor scheduleInfoProcessor) {
        this.context = context;
        this.taskListProvider = taskListProvider;
        this.interceptorsList = interceptorsList;
        this.jobExecutingMonitor = jobExecutingMonitor;
        this.scheduleInfoConverter = scheduleInfoConverter;
        this.scheduleInfoProcessor = scheduleInfoProcessor;
    }

    @PostConstruct
    public void init() {
        taskIdTaskDescription = taskListProvider.getTasks()
                .stream()
                .collect(toMap(TaskDescription::getName, taskDescription -> taskDescription));
    }

    @Override
    public TaskHooks startJob(Job job) {
        TaskDescription taskDescription = taskIdTaskDescription.get(job.taskId().name());
        if (taskDescription == null) {
            throw new IllegalStateException("Job wasn't found " + job);
        }

        Class<? extends BaseDirectJob> clazz =
                taskDescription.getTaskClass().asSubclass(BaseDirectJob.class);

        BaseDirectJob baseDirectJob = context.getBean(clazz);
        baseDirectJob.setInterceptorsList(interceptorsList);

        HourglassJob hourglassJob;

        if (clazz.isAnnotationPresent(HourglassDaemon.class)) {
            var runPeriod = clazz.getAnnotation(HourglassDaemon.class).runPeriod();
            var maxIteration = clazz.getAnnotation(HourglassDaemon.class).maxExecuteIterations();
            hourglassJob = new DaemonJobWrapper(baseDirectJob, runPeriod, maxIteration);
        } else {
            hourglassJob = new PeriodicJobWrapper(baseDirectJob);
        }

        var scheduleInfo = scheduleInfoConverter.getScheduleInfoFromEncodedString(job.getScheduleHash());
        var nextRunCalculator = scheduleInfoProcessor.getNextRunCalculator(scheduleInfo,
                taskDescription.getScheduleReducer());
        TaskParametersMapImpl paramsMap = TaskParametersMapImpl.fromString(job.taskId().param());
        return new TaskHooksImpl(hourglassJob, jobExecutingMonitor, nextRunCalculator, paramsMap);
    }

}
