package ru.yandex.direct.scheduler.hourglass.implementations;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import org.springframework.context.ApplicationContext;

import ru.yandex.direct.scheduler.HourglassDaemon;
import ru.yandex.direct.scheduler.hourglass.HourglassJob;
import ru.yandex.direct.scheduler.hourglass.ParamDescription;
import ru.yandex.direct.scheduler.hourglass.TaskParametersMap;
import ru.yandex.direct.scheduler.support.ParameterizedBy;
import ru.yandex.direct.scheduler.support.ParametersSource;

import static com.google.common.base.Preconditions.checkState;
import static ru.yandex.direct.scheduler.support.DirectParameterizedJob.PARAM_NAME;
import static ru.yandex.direct.scheduler.support.DirectParameterizedJob.getAllParams;
import static ru.yandex.direct.scheduler.support.DirectParameterizedJob.isParameterized;
import static ru.yandex.direct.scheduler.support.DirectShardedJob.SHARD_PARAM;
import static ru.yandex.direct.scheduler.support.DirectShardedJob.isShardedJobClass;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

public class TaskParameterizer {
    private final ApplicationContext context;
    private final List<String> shardIndexes;

    public TaskParameterizer(ApplicationContext context, Collection<Integer> shardIndexes) {
        this.context = context;
        this.shardIndexes = mapList(shardIndexes, Object::toString);
    }

    private Class<? extends ParametersSource> getParametersSource(Class<? extends HourglassJob> jobClass) {
        checkState(isJobParameterized(jobClass));
        return jobClass.getAnnotation(ParameterizedBy.class).parametersSource();
    }

    private boolean isJobSharded(Class<? extends HourglassJob> jobClass) {
        return isShardedJobClass(jobClass);
    }

    private boolean isJobParameterized(Class<? extends HourglassJob> jobClass) {
        return jobClass.getAnnotation(ParameterizedBy.class) != null;
    }

    private void checkParametrizedConstraints(Class<? extends HourglassJob> jobClass) {
        if (isJobParameterized(jobClass) && !isParameterized(jobClass)) {
            throw new IllegalStateException("Job " + jobClass
                    + " should inherit DirectParameterizedJob since it annotated with @ParameterizedBy");
        }
        if (!isJobParameterized(jobClass) && isParameterized(jobClass)) {
            throw new IllegalStateException("Job " + jobClass
                    + " should be annotated using @ParameterizedBy since it inherits DirectParameterizedJob");
        }
    }

    public List<ParamDescription> getAllParameters(Class<? extends HourglassJob> jobClass) {
        List<String> serializedParams;
        List<ParamDescription> descriptions = new ArrayList<>();
        checkParametrizedConstraints(jobClass);

        String paramName;

        boolean isDaemon = jobClass.isAnnotationPresent(HourglassDaemon.class);

        if (isJobParameterized(jobClass)) {
            serializedParams = getAllParams(getParametersSource(jobClass), context);
            paramName = PARAM_NAME;
        } else if (isJobSharded(jobClass)) {
            serializedParams = shardIndexes;
            paramName = SHARD_PARAM;
        } else {
            descriptions.add(new ParamDescriptionImpl(TaskParametersMap.of("is_daemon", String.valueOf(isDaemon))));
            return descriptions;
        }

        int paramNumber = 0;
        for (var serializedParam : serializedParams) {
            var description = new ParamDescriptionImpl(++paramNumber, serializedParams.size(),
                    TaskParametersMap.of(paramName, serializedParam,
                    "is_daemon", String.valueOf(isDaemon)));
            descriptions.add(description);
        }
        return descriptions;
    }

}
