package ru.yandex.direct.scheduler.hourglass.implementations;

import java.io.IOException;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import javax.annotation.Nonnull;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.type.MapType;
import com.fasterxml.jackson.databind.type.TypeFactory;

import ru.yandex.direct.scheduler.hourglass.TaskParametersMap;

import static com.fasterxml.jackson.databind.SerializationFeature.ORDER_MAP_ENTRIES_BY_KEYS;

public class TaskParametersMapImpl implements TaskParametersMap {
    private final Map<String, String> map;
    private final String mapAsString;

    private static final ObjectMapper mapper =
            new ObjectMapper().configure(ORDER_MAP_ENTRIES_BY_KEYS, true);
    private static final MapType mapType = getMapType();

    public TaskParametersMapImpl() {
        this(Map.of());
    }

    public TaskParametersMapImpl(@Nonnull Map<String, String> map) {
        this.map = Map.copyOf(map);
        var mapper = new ObjectMapper().configure(ORDER_MAP_ENTRIES_BY_KEYS, true);
        try {
            this.mapAsString = mapper.writeValueAsString(map);
        } catch (JsonProcessingException e) {
            throw new IllegalStateException("Failed to serialize parameters map " + map, e);
        }
    }

    @Override
    public String toString() {
        return "TaskParametersMapImpl{" +
                "map=" + map +
                '}';
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        TaskParametersMapImpl that = (TaskParametersMapImpl) o;
        return Objects.equals(map, that.map);
    }

    @Override
    public int hashCode() {
        return Objects.hash(map);
    }

    @Override
    public String getAsString() {
        return mapAsString;
    }

    // todo фабрика создания мапы
    public static TaskParametersMapImpl fromString(String stringParametersMap) {
        try {
            return new TaskParametersMapImpl(mapper.readValue(stringParametersMap, mapType));
        } catch (IOException e) {
            throw new IllegalArgumentException("Failed to deserialize parameter " + stringParametersMap, e);
        }
    }

    @Override
    public int size() {
        return map.size();
    }

    @Override
    public boolean isEmpty() {
        return map.isEmpty();
    }

    @Override
    public boolean containsKey(String key) {
        return map.containsKey(key);
    }

    @Override
    public String get(String key) {
        return map.get(key);
    }

    @Override
    @Nonnull
    public Set<String> keySet() {
        return map.keySet();
    }

    @Override
    @Nonnull
    public Collection<String> values() {
        return map.values();
    }

    @Override
    @Nonnull
    public Set<Map.Entry<String, String>> entrySet() {
        return map.entrySet();
    }

    private static MapType getMapType() {
        TypeFactory typeFactory = mapper.getTypeFactory();
        return typeFactory.constructMapType(HashMap.class, String.class, String.class);
    }
}
