package ru.yandex.direct.scheduler.hourglass.implementations.schedule.strategies;

import java.time.Clock;
import java.time.Instant;
import java.time.temporal.ChronoUnit;

import javax.annotation.Nonnull;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.hourglass.TaskProcessingResult;
import ru.yandex.direct.scheduler.hourglass.implementations.schedule.strategies.data.SchedulePeriodicData;
import ru.yandex.direct.scheduler.hourglass.schedule.strategies.NextRunCalcStrategy;
import ru.yandex.direct.scheduler.hourglass.schedule.strategies.ScheduleData;
import ru.yandex.direct.scheduler.hourglass.schedule.strategies.ScheduleType;

@Component
public class PeriodicStrategy implements NextRunCalcStrategy {

    private final Clock clock;

    @Autowired
    public PeriodicStrategy() {
        this(Clock.systemUTC());
    }

    public PeriodicStrategy(Clock clock) {
        this.clock = clock;
    }

    @Override
    public Instant getNextDate(@Nonnull TaskProcessingResult taskProcessingResult, ScheduleData scheduleData) {

        var schedulePeriodicData = (SchedulePeriodicData) scheduleData;
        long period = schedulePeriodicData.getInterval();

        if (period == 0) {
            return clock.instant();
        }

        long currentPeriod = (clock.instant().getEpochSecond() / period) * period;

        long nextRunInSeconds = currentPeriod; //By default schedule a next run to current period

        if (taskProcessingResult.lastStartTime() != null) {
            long lastStartTime =
                    taskProcessingResult.lastStartTime().truncatedTo(ChronoUnit.SECONDS).getEpochSecond();

            if (lastStartTime >= currentPeriod) { //Have already run in the current period
                nextRunInSeconds += period;
            }
        }
        return Instant.ofEpochSecond(nextRunInSeconds);
    }

    @Override
    public ScheduleType getType() {
        return ScheduleType.PERIODIC;
    }
}
