package ru.yandex.direct.scheduler.support;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.scheduler.JobInterceptorsList;
import ru.yandex.direct.scheduler.hourglass.HourglassJob;
import ru.yandex.direct.scheduler.hourglass.TaskParametersMap;

import static com.google.common.base.Preconditions.checkState;

/**
 * Базовый класс для всех Директовских задач.
 * <p>
 * Каждой задаче доступны следующие методы:
 * <ul>
 * <li>{@link #setJugglerStatus(JugglerStatus, String)} - задать на текущей итерации выполнения
 * статус задачи (для Juggler-мониторинга) и его описание</li>
 * <li>{@link TaskParametersMap} {@link #getParametersMap()} - контекст выполнения задачи</li>
 * <li>{@link JugglerStatus} {@link #getJugglerStatus()} - получить статус задачи на текущей итерации</li>
 * <li>{@link String} {@link #getJugglerDescription()} - получить описание статуса задачи</li>
 * </ul>
 * <p>
 * Типовой жизненный цикл задачи:
 * <pre>{@code
 *      BaseDirectJob job = context.getBean(Class<? extends BaseDirectJob>);
 *      job.setInterceptorsList(JobInterceptorsList);
 *      job.execute(JobExecutionContext);
 * }</pre>
 * <p>
 * Особенности реализации:
 * <ul>
 * <li>{@link #execute()} запоминает контекст только при первом вызове, при повторных вызовах
 * переданный контекст игнорируется</li>
 * <li>вызов {@link #setInterceptorsList(JobInterceptorsList)} возможен только до первого вызова
 * {@link #execute()}</li>
 * <li>вызов {@link #run()} возможен только после вызова {@link #execute()}</li>
 * <li>задача оборачивается в интерсепторы при каждом вызове {@link #execute()},
 * однако в будущем это поведение может измениться</li>
 * </ul>
 */
@ParametersAreNonnullByDefault
public abstract class BaseDirectJob implements WrappedJob, HourglassJob {
    private static final Logger logger = LoggerFactory.getLogger(BaseDirectJob.class);

    private static final JugglerStatus defaultJugglerStatus = JugglerStatus.OK;
    private static final String defaultJugglerDescription = "job finished";

    private static final JobInterceptorsList EMPTY_INTERCEPTORS_LIST = new JobInterceptorsList();
    private boolean initialized = false;

    private TaskParametersMap context;
    private JobInterceptorsList interceptorsList = EMPTY_INTERCEPTORS_LIST;

    private JugglerStatus jugglerStatus;
    private String jugglerDescription;

    /**
     * После перегрузки выполняет работу джоба.
     */
    public abstract void execute();

    /**
     * Запускает задачу. Инициализация и очистка происходят в классах обертках {@link DaemonJobWrapper}
     * {@link PeriodicJobWrapper}
     *
     * @see DaemonJobWrapper#execute(TaskParametersMap)
     */
    @Override
    public void execute(TaskParametersMap taskParametersMap) {
        interceptorsList.wrap(this).run();
    }

    /**
     * Запускает graceful degradation задачи. Необходимая реализация переопределяется в конкретном джобе.
     */
    @Override
    public void onShutdown() {
        logger.debug("Call on Job's empty default graceful degradation method");
    }

    /**
     * Метод вызывается JobWrapper'ами.
     * Проверяет, что задача инициализирована контекстом и вызывает {@link #execute()}
     *
     * @throws IllegalStateException в случае если задача не была инициализирована
     */
    public final void run() {
        checkState(initialized, "Job must be initialized before execute");
        execute();
    }

    /**
     * Позволяет реализовать подготовку данных, например получение параметров задачи из контекста.
     *
     * @see DirectShardedJob
     */
    void prepare() {
    }

    /**
     * Вызывается освобождения ресурсов после выполнения джоба
     */
    public void finish() {
    }

    public void setInterceptorsList(JobInterceptorsList interceptorsList) {
        checkState(!initialized, "Changing InterceptorsList is forbidden after initialization");
        this.interceptorsList = interceptorsList;
    }

    /**
     * Инициализировать задачу.
     * Сохраняет контекст задачи, выполняет {@link #prepare()}.
     *
     * @param taskParametersMap параметры выполнения задачи
     */
    public void initialize(TaskParametersMap taskParametersMap) {
        checkState(!initialized, "initialize method can only be called once.");
        this.context = taskParametersMap;
        initialized = true;
        prepare();
    }

    /**
     * Получить контекст выполнения задачи
     *
     * @return контекст задачи, переданный при первом вызове {@link #execute()}
     * @throws IllegalStateException если задача не была инициализирована
     */
    public TaskParametersMap getParametersMap() {
        checkState(initialized, "Job must be initialized");
        return context;
    }

    /**
     * Получить Juggler-статус задачи из контекста.
     *
     * @return последний заданный явно статус или умолчание {@link #defaultJugglerStatus}.
     */
    public JugglerStatus getJugglerStatus() {
        return jugglerStatus != null ? jugglerStatus : defaultJugglerStatus;
    }

    /**
     * Получить описание Juggler-статуса задачи.
     *
     * @return последний заданный явно статус или умолчание {@link #defaultJugglerDescription}.
     */
    public String getJugglerDescription() {
        return jugglerDescription != null ? jugglerDescription : defaultJugglerDescription;
    }

    /**
     * Задать Juggler-статус задачи и описание к нему.
     */

    public void setJugglerStatus(@Nullable JugglerStatus status, @Nullable String description) {
        jugglerStatus = status;
        jugglerDescription = description;
    }
}
