package ru.yandex.direct.scheduler.support;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.scheduler.hourglass.HourglassInterruptibleJob;
import ru.yandex.direct.scheduler.hourglass.TaskParametersMap;

/**
 * Обёртка для джоба, превращающая джоб в демон
 * <p>
 * Оборачивает джоб, собственно выполняющий полезную работу, и запускает его по предоставленному таймеру,
 * попутно обрабатывая прерывания от шедулера и системы.
 * <p>
 * Implementation Note:
 * <p>
 * строится на основе аннотаций в
 * {@link DirectJob}/{@link DirectShardedJob}
 */
@ParametersAreNonnullByDefault
public class DaemonJobWrapper implements HourglassInterruptibleJob {
    private static final Logger logger = LoggerFactory.getLogger(DaemonJobWrapper.class);

    private final BaseDirectJob job;
    private final InterruptableTimer timer;
    private final int periodInSeconds;
    private final int maxIterations;
    private volatile boolean interrupted = false;


    /**
     * Создание над Job-ом обёртки со вкусом Директа
     *
     * @param job             Джоб, выполняющий полезную работу
     * @param periodInSeconds Период запуска итераций, к секундах
     * @param maxIterations   макстимальное количество итераций демона
     */
    public DaemonJobWrapper(BaseDirectJob job, int periodInSeconds, int maxIterations) {
        this(job, new InterruptableTimer(), periodInSeconds, maxIterations);
    }

    DaemonJobWrapper(BaseDirectJob job, InterruptableTimer timer, int periodInSeconds,
                     int maxIterations) {
        this.job = job;
        this.timer = timer;
        this.periodInSeconds = periodInSeconds;
        this.maxIterations = maxIterations;
    }

    /**
     * Инициализирует, запускает и очищает демон
     *
     * @param context Данные запуска, содержащие, в частности, номер шарда
     */
    @Override
    public void execute(TaskParametersMap context) {
        job.initialize(context);
        int currentIteration = 0;
        logger.debug("daemon job started");
        timer.set(periodInSeconds);
        try {
            while (!interrupted && currentIteration < maxIterations) {
                job.execute(context);
                currentIteration++;
                cleanupBetweenIterations();
                timer.await();
            }
        } finally {
            job.finish();
        }
        logger.debug("daemon job finished");
    }

    /**
     * Запускает graceful degradation задачи.
     */
    @Override
    public void onShutdown() {
        interruptTimer();
        try {
            job.onShutdown();
        } catch (RuntimeException e) {
            logger.error("Error during graceful shutdown of daemon job", e);
        }
    }

    /**
     * Прерывает исполнение демона
     */
    @Override
    public void interrupt() {
        interruptTimer();
    }

    private void interruptTimer() {
        interrupted = true;
        timer.interrupt();
    }

    private void cleanupBetweenIterations() {
        job.setJugglerStatus(null, null);
    }
}
