package ru.yandex.direct.scheduler.support;

import java.util.List;

import org.springframework.context.ApplicationContext;

import ru.yandex.direct.scheduler.hourglass.TaskParametersMap;

import static com.google.common.base.Preconditions.checkState;
import static java.util.stream.Collectors.toList;

/**
 * Базовый класс для параметризованных задач.
 * Идея на будущее: можно попробовать ограничить TParam некоторым интерфейсом, который сам умеет
 * выполнять сериализацию-десериализацию объекта-параметра, вместо того, чтобы заставлять это делать
 * ParametersSource.
 */
public abstract class DirectParameterizedJob<TParam> extends BaseDirectJob {
    public static final String PARAM_NAME = "param";

    private String param;

    @Override
    void prepare() {
        TaskParametersMap context = getParametersMap();

        checkState(context.containsKey(PARAM_NAME),
                "JobExecutionContext should contains '%s' parameter for parameterized job)", PARAM_NAME);

        this.param = context.get(PARAM_NAME);
    }

    /**
     * Возвращает строковое значение параметра, назначенного задаче
     */
    public String getParam() {
        return param;
    }

    public static boolean isParameterized(Class<?> jobClass) {
        return DirectParameterizedJob.class.isAssignableFrom(jobClass);
    }

    /**
     * Извлекает все возможные значения параметров (сразу в сериализованном в строку виде)
     * из бина с классом parametersSourceClazz
     */
    public static List<String> getAllParams(Class<? extends ParametersSource> parametersSourceClazz,
                                            ApplicationContext appContext) {
        ParametersSource parametersSource = appContext.getBean(parametersSourceClazz);
        //noinspection unchecked
        List<Object> allParamValues = parametersSource.getAllParamValues();
        //noinspection Convert2MethodRef,unchecked
        return allParamValues.stream()
                .map(v -> parametersSource.convertParamToString(v))
                .collect(toList());
    }
}
