package ru.yandex.direct.scheduler.support;

import ru.yandex.direct.scheduler.hourglass.TaskParametersMap;

import static com.google.common.base.Preconditions.checkState;

/**
 * Базовый класс для всех шардированных джобов Директа.
 * <p>
 * Дополнительно делает доступными:
 * <ul>
 * <li>{@code int} {@link #getShard()} - обрабатываемый задачей номер шарда</li>
 * </ul>
 *
 * @see BaseDirectJob
 */
public abstract class DirectShardedJob extends BaseDirectJob {
    public static final String SHARD_PARAM = "shard";

    private static final int NO_SHARD = Integer.MIN_VALUE;
    private int shard = NO_SHARD;

    /**
     * Конструктор для использования из конструкторов для тестов, позволяет задать номер шарда
     *
     * @param shard номер шарда
     */
    protected DirectShardedJob(int shard) {
        this.shard = shard;
    }

    /**
     * Только для тестов
     */
    public DirectShardedJob withShard(int shard) {
        this.shard = shard;
        return this;
    }

    /**
     * Конструктор по умолчанию
     */
    public DirectShardedJob() {
    }

    /**
     * Определить, является ли класс - "шардированной задачей".
     * В реальности проверяется, что переданный класс наследник от текущего.
     *
     * @param clazz класс для проверки
     */
    public static boolean isShardedJobClass(Class clazz) {
        return DirectShardedJob.class.isAssignableFrom(clazz);
    }

    /**
     * Задает значение {@link #shard}, доставая его из {@link #context} (кладется туда движком Кварца,
     * определяется при задании расписания джоба)
     *
     * @throws IllegalStateException если значения {@link #SHARD_PARAM} не оказалось в контексте задачи
     */
    @Override
    final void prepare() {
        TaskParametersMap context = getParametersMap();

        checkState(context.containsKey(SHARD_PARAM),
                "JobExecutionContext should contains '%s' parameter for sharded job)", SHARD_PARAM);

        this.shard = Integer.parseInt(context.get(SHARD_PARAM));
    }

    /**
     * Получить номер шарда, для которого выполняется задача
     *
     * @return номер шарда
     * @throws IllegalStateException если шард не был задан
     */
    public int getShard() {
        checkState(shard != NO_SHARD, "Job must be initialized before getShard");
        return shard;
    }
}
