package ru.yandex.direct.scheduler.support;

import java.util.concurrent.CountDownLatch;
import java.util.function.Supplier;

import static java.util.concurrent.TimeUnit.MILLISECONDS;

/**
 * Прерываемый таймер
 */
public class InterruptableTimer {
    private int periodInSeconds;
    private Long lastFireTimeMills;
    private final CountDownLatch interrupted;
    private final Supplier<Long> nowMillsGetter;

    InterruptableTimer(CountDownLatch interrupted, Supplier<Long> nowMillsGetter) {
        this.interrupted = interrupted;
        this.nowMillsGetter = nowMillsGetter;
    }

    public InterruptableTimer() {
        this(new CountDownLatch(1), () -> System.nanoTime() / 1_000_000);
    }

    /**
     * Устанавливает таймер - выставляет период срабатывания и время начала отсчета
     *
     * @param seconds период срабатывания в секундах
     */
    public void set(int seconds) {
        periodInSeconds = seconds;
        lastFireTimeMills = nowMillsGetter.get();
    }

    /**
     * Ожидает ближайшего срабатывания таймера
     */
    public void await() {
        if (lastFireTimeMills == null) {
            throw new IllegalStateException("Timer is not set.");
        }

        long now = nowMillsGetter.get();
        long workingTime = (now - lastFireTimeMills);
        long timeLeftToSleep = Math.max(periodInSeconds * 1000L - workingTime, 0);

        try {
            //noinspection ResultOfMethodCallIgnored
            interrupted.await(timeLeftToSleep, MILLISECONDS);
            lastFireTimeMills = now + timeLeftToSleep;
        } catch (InterruptedException ex) {
            Thread.currentThread().interrupt();
        }
    }

    /**
     * Прерывает ожидание таймера
     */
    public void interrupt() {
        interrupted.countDown();
    }
}
