package ru.yandex.direct.scheduler.support;

import java.time.Duration;
import java.util.concurrent.atomic.AtomicLong;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;

import ru.yandex.direct.juggler.JugglerEvent;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.scheduler.hourglass.implementations.monitoring.juggler.HourglassJugglerSender;

import static com.google.common.base.Preconditions.checkState;
import static org.springframework.beans.factory.config.ConfigurableBeanFactory.SCOPE_PROTOTYPE;
import static ru.yandex.direct.scheduler.hourglass.implementations.monitoring.juggler.HourglassJugglerSender.getPrefixedService;

/**
 * Отправка в мониторинг сигнала о том, что задачи были запущены
 * Чтобы не заспамить juggler - отправляет события не чаще, чем раз каждые {@code interval} времени.
 */
@ParametersAreNonnullByDefault
@Component
@Scope(SCOPE_PROTOTYPE)
public class JobExecutingMonitor {
    private static final Logger logger = LoggerFactory.getLogger(JobExecutingMonitor.class);
    public static final String JOBS_EXECUTED_SERVICE = getPrefixedService("jobs_executed");

    private final AtomicLong executedJobs = new AtomicLong(0L);
    private final HourglassJugglerSender jugglerSender;

    private String schedulerName = "unknown";
    private long intervalInNanos = Duration.ofSeconds(1).toNanos();
    private long lastSent = 0;

    @Autowired
    public JobExecutingMonitor(HourglassJugglerSender jugglerSender) {
        this.jugglerSender = jugglerSender;
    }

    public void setSchedulerName(String schedulerName) {
        this.schedulerName = schedulerName;
    }

    public void setInterval(Duration interval) {
        checkState(!interval.isNegative(), "interval couldn't be negative");
        this.intervalInNanos = interval.toNanos();
    }

    public String getName() {
        return this.getClass().getName();
    }

    public void jobWasExecuted() {
        executedJobs.incrementAndGet();

        if (System.nanoTime() - lastSent > intervalInNanos) {
            try {
                sendSignal();
            } catch (RuntimeException e) {
                logger.warn("Failed to send jobs executed signal to juggler", e);
            }
        }
    }

    private void sendSignal() {
        long executed = executedJobs.getAndSet(0);
        jugglerSender.sendEventSafely(new JugglerEvent(JOBS_EXECUTED_SERVICE, JugglerStatus.OK,
                "Executed " + executed + " jobs"));
        lastSent = System.nanoTime();
    }
}
