package ru.yandex.direct.scheduler.support;

import java.util.List;

/**
 * Штука, которая умеет получать значения параметров для параметризованных джобов.
 * Может быть использована, если нужно параллельно запускать сколько-то джобов, каждая из которых
 * занимается своей частью данных (своим шардом или кластером Yt).
 * Например, таким образом можно создать джобы, работающие каждый со своим кластером SENECA.
 * Для этого достаточно создать Spring-компонент, который знает номера активных кластеров, реализовать в нём
 * этот интерфейс, а в джобах указать класс этого компонента через аннотацию {@link ParameterizedBy}.
 * <p>
 * Пример для шардов PPC:
 * <pre>
 * {@literal @}Component
 *  public class PpcShardsParametersSource implements ParametersSource {
 *    private final ShardHelper shardHelper;
 *
 *   {@literal @}Autowired
 *    public PpcShardsParametersSource(ShardHelper shardHelper) {
 *      this.shardHelper = shardHelper;
 *    }
 *
 *   {@literal @}Override
 *    public List<Integer> getAllParamValues() {
 *      return new ArrayList<>(shardHelper.dbShards());
 *    }
 *
 *    {@literal @}Override
 *    public String convertParamToString(Integer paramValue) {
 *      return paramValue.toString();
 *    }
 *
 *    {@literal @}Override
 *    public Integer convertStringToParam(String string) {
 *      return Integer.parseInt(string);
 *    }
 *  }
 * </pre>
 * <p>
 * И использование его в джобе:
 * <pre>
 * {@literal @}ParameterizedBy(parametersSource = PpcShardsParametersSource.class)
 *  public class MyShardedPpcJob extends DirectParameterizedJob<Integer> {
 *  }
 * </pre>
 */
public interface ParametersSource<TParam> {
    /**
     * Возвращает все возможные значения параметров.
     * Вызывается перед назначением расписания (триггеров) задаче.
     */
    List<TParam> getAllParamValues();

    /**
     * Конвертирует значение параметра в строчку.
     * Эта строчка будет записана кварцем как в data джобы, так и в её name и в name триггера.
     * Строка должна помещаться в 200 символов вместе с именем джобы.
     */
    String convertParamToString(TParam paramValue);

    /**
     * Метод, осуществляющий обратное преобразование.
     * Позволяет по строчке, извлечённой кварцем перед запуском джобы, понять, какой параметр был ей передан.
     */
    TParam convertStringToParam(String string);
}
