package ru.yandex.direct.solomon;

import java.util.EnumSet;
import java.util.Set;

import javax.annotation.Nullable;

import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.primitives.Rate;

import static ru.yandex.direct.solomon.SolomonResponseMonitorStatus.STATUS_2XX;
import static ru.yandex.direct.solomon.SolomonResponseMonitorStatus.STATUS_4XX;
import static ru.yandex.direct.solomon.SolomonResponseMonitorStatus.STATUS_5XX;
import static ru.yandex.direct.solomon.SolomonUtils.EXTERNAL_SYSTEM_LABEL_NAME;
import static ru.yandex.direct.solomon.SolomonUtils.incrementIfNotNull;

public class SolomonExternalSystemMonitorService {
    public static final String REQS_COUNT_SENSOR_NAME = "reqs.count";
    public static final String INTERPRETED_STATUS_LABEL_NAME = "interpreted_status";
    private final String externalSystem;
    private final Set<String> allowedMethods;

    /**
     * Метрики для статусов
     * {@link SolomonResponseMonitorStatus#STATUS_2XX},
     * {@link SolomonResponseMonitorStatus#STATUS_4XX},
     * {@link SolomonResponseMonitorStatus#STATUS_5XX},
     * будут проинициализованы нулями для методов, перечисленных в {@code allowedMethods}
     */
    public SolomonExternalSystemMonitorService(
            String externalSystem,
            Set<String> allowedMethods
    ) {
        this.externalSystem = externalSystem;
        this.allowedMethods = allowedMethods;
        initStatusMetrics(allowedMethods, EnumSet.of(STATUS_2XX, STATUS_4XX, STATUS_5XX));
    }

    /**
     * Проинициализировать метрики для статусов {@code statuses} нулями для методов,
     * перечисленных в {@code allowedMethods}
     */
    private void initStatusMetrics(Set<String> allowedMethods, EnumSet<SolomonResponseMonitorStatus> statuses) {
        for (String method : allowedMethods) {
            for (var status : statuses) {
                //noinspection ConstantConditions
                getMetric(method, status).add(0);
            }
        }
    }

    @Nullable
    private Rate getMetric(String methodName, SolomonResponseMonitorStatus status) {
        if (!allowedMethods.contains(methodName)) {
            return null;
        }

        Labels labels = Labels.of(
                EXTERNAL_SYSTEM_LABEL_NAME, externalSystem,
                "method", methodName,
                INTERPRETED_STATUS_LABEL_NAME, status.getName()
        );
        return SolomonUtils.getExternalRateMetric(REQS_COUNT_SENSOR_NAME, labels);
    }

    /**
     * Записать получение статуса {@param status} у метода {@param methodName}
     */
    public void write(String methodName, SolomonResponseMonitorStatus status) {
        Rate rate = getMetric(methodName, status);
        incrementIfNotNull(rate);
    }
}
