package ru.yandex.direct.solomon;

import java.time.Duration;
import java.util.Arrays;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;

import javax.annotation.Nonnull;
import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import com.google.common.collect.ImmutableSet;
import com.google.common.net.HttpHeaders;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.monlib.metrics.encode.MetricEncoder;
import ru.yandex.monlib.metrics.encode.MetricFormat;
import ru.yandex.monlib.metrics.encode.json.MetricJsonEncoder;
import ru.yandex.monlib.metrics.encode.spack.MetricSpackEncoder;
import ru.yandex.monlib.metrics.encode.spack.format.CompressionAlg;
import ru.yandex.monlib.metrics.encode.spack.format.TimePrecision;

/**
 * Сервлет, отдающий "Соломону" данные по сенсорам по запросу
 */
public class SolomonMonitoringServlet extends HttpServlet {
    private static final Logger logger = LoggerFactory.getLogger(SolomonMonitoringServlet.class);
    private static final int SEMAPHORE_LIMIT = 2;

    private static final MetricFormat DEFAULT_FORMAT = MetricFormat.JSON;
    private static final CompressionAlg DEFAULT_COMPRESSION = CompressionAlg.NONE;
    private static final String FORMAT_PARAMETER = "format";

    private static final Duration SEMAPHORE_ACQUIRE_TIMEOUT = Duration.ofSeconds(2);

    private static final ImmutableSet<MetricFormat> SUPPORTED_FORMATS =
            ImmutableSet.of(MetricFormat.JSON, MetricFormat.SPACK);

    private static final Map<String, Consumer<MetricEncoder>> REGISTRIES = Map.of(
            "default", SolomonUtils::dump,
            "external-metrics", SolomonUtils::dumpExternalMetrics,
            "bs-export-queue-metrics", SolomonUtils::dumpBsExportQueueMetrics,
            "mod-export-queue-metrics", SolomonUtils::dumpModExportQueueMetrics,
            "common-metrics", SolomonUtils::dumpCommonMetrics,
            "frontend-timings-metrics", SolomonUtils::dumpFrontendTimingMetrics,
            "trace-log-functions-metrics", SolomonUtils::dumpTraceLogFunctionsMetrics,
            "unified-agent-client-metrics", SolomonUtils::dumpUnifiedAgentClientMetrics,
            "grut-object-api-client-metrics", SolomonUtils::dumpGrutObjectApiMetrics
    );
    private static final ConcurrentHashMap<String, Semaphore> SEMAPHORES = new ConcurrentHashMap<>();

    @Override
    protected void doGet(HttpServletRequest request, HttpServletResponse response) {
        Semaphore semaphore;
        Consumer<MetricEncoder> registry;
        String requestedRegistry = request.getParameter("registry");
        switch (request.getRequestURI()) {
            case "/monitoring":
                if (requestedRegistry == null) {
                    requestedRegistry = "default";
                }
                break;
            case "/external-metrics":
                requestedRegistry = "external-metrics";
                break;
            case "/common-metrics":
                requestedRegistry = "common-metrics";
                break;
            case "/frontend-timings-metrics":
                requestedRegistry = "frontend-timings-metrics";
                break;
            case "/trace-log-functions-metrics":
                requestedRegistry = "trace-log-functions-metrics";
                break;
            case "/unified-agent-client-metrics":
                requestedRegistry = "unified-agent-client-metrics";
                break;
            case "/grut-object-api-client-metrics":
                requestedRegistry = "grut-object-api-client-metrics";
                break;
            default:
                logger.error("request URI {} is not serviced", request.getRequestURI());
                response.setStatus(HttpServletResponse.SC_NOT_FOUND);
                return;
        }

        if (!REGISTRIES.containsKey(requestedRegistry)) {
                logger.error("request registry {} is not serviced", requestedRegistry);
                response.setStatus(HttpServletResponse.SC_NOT_FOUND);
                return;
        }
        registry = REGISTRIES.get(requestedRegistry);
        semaphore = SEMAPHORES.computeIfAbsent(requestedRegistry, key -> new Semaphore(SEMAPHORE_LIMIT));

        try {
            if (semaphore.tryAcquire(SEMAPHORE_ACQUIRE_TIMEOUT.toNanos(), TimeUnit.NANOSECONDS)) {
                try (
                        ServletOutputStream out = response.getOutputStream();
                        MetricEncoder encoder = prepareEncoder(request, response, out);
                ) {
                    registry.accept(encoder);
                    out.flush();
                } finally {
                    semaphore.release();
                }
            } else {
                logger.error("Semaphore is already acquired");
                response.setStatus(429);
            }
        } catch (Exception e) {
            logger.error("Error while write Solomon sensors", e);
            response.setStatus(HttpServletResponse.SC_INTERNAL_SERVER_ERROR);
        }
    }

    private MetricEncoder prepareEncoder(HttpServletRequest request, HttpServletResponse response,
                                          ServletOutputStream out) {
        MetricFormat format = findFormat(request);

        response.setStatus(HttpServletResponse.SC_OK);
        if (format == MetricFormat.SPACK) {
            CompressionAlg compression = findCompression(request);
            response.setContentType(format.contentType());
            return new MetricSpackEncoder(
                    TimePrecision.SECONDS,
                    compression,
                    out
            );
        } else if (format == MetricFormat.JSON) {
            response.setContentType(format.contentType());
            return new MetricJsonEncoder(out);
        } else {
            throw new IllegalStateException("Unsupported format: " + format);
        }
    }

    @Nonnull
    private MetricFormat findFormat(HttpServletRequest request) {
        String format = request.getParameter(FORMAT_PARAMETER);
        if (format == null) {
            format = request.getHeader(HttpHeaders.ACCEPT);
            if (format == null) {
                return DEFAULT_FORMAT;
            }
        }

        return Arrays.stream(format.split(","))
                .map(MetricFormat::byContentType)
                .filter(SUPPORTED_FORMATS::contains)
                .findFirst()
                .orElse(DEFAULT_FORMAT);
    }

    @Nonnull
    private CompressionAlg findCompression(HttpServletRequest request) {
        String compression = request.getHeader(HttpHeaders.ACCEPT_ENCODING);
        if (compression == null) {
            return DEFAULT_COMPRESSION;
        }
        return Arrays.stream(compression.split(","))
                .map(CompressionAlg::byEncoding)
                .filter(x -> x != CompressionAlg.NONE)
                .findFirst()
                .orElse(DEFAULT_COMPRESSION);
    }
}
