package ru.yandex.direct.solomon;

import java.io.ByteArrayOutputStream;

import org.asynchttpclient.RequestBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.asynchttp.ParallelFetcher;
import ru.yandex.direct.asynchttp.ParallelFetcherFactory;
import ru.yandex.direct.asynchttp.ParsableStringRequest;
import ru.yandex.direct.asynchttp.Result;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.utils.InterruptedRuntimeException;
import ru.yandex.monlib.metrics.encode.json.MetricJsonEncoder;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.registry.MetricRegistry;

import static java.nio.charset.StandardCharsets.UTF_8;

/**
 * микроимплементация клиента для пуша метрик в Solomon / Solomon-agent
 */
public class SolomonPushClient {
    private static final Logger logger = LoggerFactory.getLogger(SolomonPushClient.class);

    private final String url;
    private final ParallelFetcherFactory parallelFetcherFactory;

    // localhost:19090/direct/push-monitoring
    public SolomonPushClient(String url, ParallelFetcherFactory parallelFetcherFactory) {
        this.url = url;
        this.parallelFetcherFactory = parallelFetcherFactory;
    }

    /**
     * отправить метрики из реестра в Solomon
     */
    public void sendMetrics(MetricRegistry registry) {
        sendMetrics(registry, System.currentTimeMillis());
    }

    /**
     * отправить метрики из реестра в Solomon
     */
    public void sendMetrics(MetricRegistry registry, long timestampMillis) {
        var data = serializeMetrics(registry, timestampMillis);
        logger.debug("Push to Solomon: {}", data);

        var req = new ParsableStringRequest(
                new RequestBuilder()
                        .setUrl(url)
                        .setMethod("POST")
                        .setHeader("Content-Type", "application/json")
                        .setBody(data)
                        .build()
        );
        ParallelFetcher<String> parallelFetcher = parallelFetcherFactory.getParallelFetcher();
        try (var ignored = Trace.current().profile("solomon:push", "", registry.estimateCount())) {
            Result<String> result = parallelFetcher.execute(req);
            checkFetcherResult(result);
        } catch (InterruptedException ex) {
            Thread.currentThread().interrupt();
            throw new InterruptedRuntimeException(ex);
        }
    }

    private void checkFetcherResult(Result<String> result) {
        if (result.getErrors() != null) {
            var errors = result.getErrors();
            var ex = new SolomonPushClientException("http request error", errors.get(errors.size() - 1));
            for (int i = 0; i < errors.size() - 1; i++) {
                ex.addSuppressed(errors.get(i));
            }
            throw ex;
        }
    }

    private String serializeMetrics(MetricRegistry registry, long timestamp) {
        var buf = new ByteArrayOutputStream();
        var encoder = new MetricJsonEncoder(buf);

        encoder.onStreamBegin(-1);
        encoder.onCommonTime(timestamp);
        registry.append(0, Labels.empty(), encoder);
        encoder.onStreamEnd();

        encoder.close();
        return buf.toString(UTF_8);
    }
}
