package ru.yandex.direct.solomon;

import java.lang.management.ManagementFactory;
import java.util.Collection;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.IntStream;

import javax.annotation.Nullable;
import javax.management.AttributeList;
import javax.management.InstanceNotFoundException;
import javax.management.MBeanServer;
import javax.management.MalformedObjectNameException;
import javax.management.ObjectName;
import javax.management.ReflectionException;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheStats;

import ru.yandex.direct.env.Environment;
import ru.yandex.direct.utils.SystemUtils;
import ru.yandex.monlib.metrics.JvmGc;
import ru.yandex.monlib.metrics.JvmMemory;
import ru.yandex.monlib.metrics.JvmRuntime;
import ru.yandex.monlib.metrics.JvmThreads;
import ru.yandex.monlib.metrics.MetricConsumer;
import ru.yandex.monlib.metrics.encode.MetricEncoder;
import ru.yandex.monlib.metrics.histogram.HistogramCollector;
import ru.yandex.monlib.metrics.histogram.Histograms;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.primitives.GaugeDouble;
import ru.yandex.monlib.metrics.primitives.GaugeInt64;
import ru.yandex.monlib.metrics.primitives.Rate;
import ru.yandex.monlib.metrics.registry.MetricRegistry;


/**
 * Содержит в себе регистр сенсоров и функции, регистрирующие сенсоры
 */
public class SolomonUtils {
    public static final MetricRegistry SOLOMON_REGISTRY = new MetricRegistry(
            getCommonLabels()
    );

    public static final String ENV_LABEL_NAME = "env";
    public static final String HOST_LABEL_NAME = "host";
    public static final String EXTERNAL_SYSTEM_LABEL_NAME = "external_system";
    public static final String SUB_SYSTEM_LABEL_NAME = "sub_system";

    /**
     * Отдельный регистр Solomon - для сенсоров внешних (относящихся не только к текущему приложению) метрик.
     * Забирается через pull механизм в отдельный сервис.
     */
    private static final MetricRegistry EXTERNAL_METRICS_REGISTRY = new MetricRegistry(getCommonLabels());

    /**
     * Отдельный регистр Solomon - для отселения метрик от остальных
     */
    public static final MetricRegistry FRONTEND_TIMINGS_METRICS_REGISTRY = new MetricRegistry(getCommonLabels());

    /**
     * Отдельный регистр Solomon - для сенсоров, которые создаются в SystemMonitoringJob
     * Забирается через pull механизм.
     */
    public static final MetricRegistry LIFELINE_METRICS_REGISTRY = new MetricRegistry(getCommonLabels());

    /**
     * Для метрик про очередь старого транспорта в БК
     */
    public static final MetricRegistry BS_EXPORT_QUEUE_REGISTRY = new MetricRegistry((getCommonLabels()));

    /**
     * Для метрик про очередь старого транспорта на модерацию
     */
    public static final MetricRegistry MOD_EXPORT_QUEUE_REGISTRY = new MetricRegistry(getCommonLabels());

    /**
     * Отдельный регистр Solomon - для trace метрик фунцкций
     */
    public static final MetricRegistry TRACE_LOG_FUNCTIONS_METRICS_REGISTRY = new MetricRegistry((getCommonLabels()));

    /**
     * Отдельный регистр Solomon для метрик Grut ObjectAPI - для отселения метрик от остальных
     */
    public static final MetricRegistry GRUT_METRICS_REGISTRY = new MetricRegistry(getCommonLabels());

    /**
     * Наборы клиентских метрик для appender'ов log4j, пишущих в Unified Agent
     * Ключа -- appenderName, значение -- набор метрик для этого аппендера
     */
    public static final Map<String, MetricRegistry> UNIFIED_AGENT_CLIENT_METRICS_REGISTRIES = new ConcurrentHashMap<>();

    /*
     * NB: поскольку продолжают появляться новые реестры — возможно их стоит хранить map'ой,
     * добавляя сразу в нее и в отдающий сервлет.
     */

    private static final Map<String, Runnable> TO_RUN_SENSORS_FUNCTIONS = new ConcurrentHashMap<>();

    private static final MBeanServer BEAN_SERVER = ManagementFactory.getPlatformMBeanServer();

    private SolomonUtils() {
        //utility class
    }

    public static int getExternalMetricsRegistryEstimateCount() {
        return EXTERNAL_METRICS_REGISTRY.estimateCount();
    }

    public static MetricRegistry newPushRegistry(Labels labels) {
        return new MetricRegistry(
                getCommonPushLabels()
                        .addAll(labels)
        );
    }

    public static MetricRegistry newPushRegistry() {
        return newPushRegistry(Labels.empty());
    }

    public static MetricRegistry newPushRegistry(String key, String value) {
        return newPushRegistry(Labels.of(key, value));
    }

    public static MetricRegistry newPushRegistry(String key1, String value1, String key2, String value2) {
        return newPushRegistry(Labels.of(key1, value1, key2, value2));
    }

    public static MetricRegistry getParallelFetcherMetricRegistry(String value) {
        return SOLOMON_REGISTRY.subRegistry("parallel_fetcher", value);
    }

    public static void addJvmMetrics() {
        addJvmMetrics(SOLOMON_REGISTRY);
    }

    public static void addJvmMetrics(MetricRegistry metricRegistry) {
        JvmGc.addMetrics(metricRegistry);

        JvmRuntime.addMetrics(metricRegistry);

        JvmThreads.addMetrics(metricRegistry);

        JvmMemory.addMetrics(metricRegistry);
    }

    public static void addUnifiedAgentLog4jMetrics(Map<String, MetricRegistry> metricRegistries) {
        UNIFIED_AGENT_CLIENT_METRICS_REGISTRIES.putAll(metricRegistries);
    }

    /**
     * Добавляет функции, которые необходимо выполнить перед записью значений сенсоров в "Соломон"
     * Предполагается, что в {@link SolomonUtils#TO_RUN_SENSORS_FUNCTIONS} находится отображение
     * идентификатора в функцию, пишущую в сенсоры метрики, которые не могут быть подсчитаны лениво
     *
     * @param id       - идентификатор функции
     * @param function - функция, которая будет выполнена перед обходом "Соломоном" сенсоров
     */
    public static void putToRunSensorFunction(String id, Runnable function) {
        TO_RUN_SENSORS_FUNCTIONS.put(id, function);
    }

    /**
     * Получить rate-метрику в отдельном регистре.
     * <p>
     * RATE-тип подходит для измерения количества сделанных запросов или отправленных объектов.
     * Не подходит для отправки "текущих значений", таких как: uptime, размер или возраст очереди.
     *
     * @param name   имя сенсора
     * @param labels метки
     * @return существующая или новая rate-метрика
     */
    public static Rate getExternalRateMetric(String name, Labels labels) {
        return EXTERNAL_METRICS_REGISTRY.rate(name, labels);
    }

    /**
     * Запись значений сенсоров в {@param out}
     *
     * @param encoder - Encoder, который пишет значения регистров
     */
    public static void dump(MetricEncoder encoder) {
        for (Runnable function : TO_RUN_SENSORS_FUNCTIONS.values()) {
            function.run();
        }

        SOLOMON_REGISTRY.supply(0, encoder);
    }

    public static void dumpExternalMetrics(MetricEncoder encoder) {
        EXTERNAL_METRICS_REGISTRY.supply(0, encoder);
    }

    public static void dumpCommonMetrics(MetricEncoder encoder) {
        LIFELINE_METRICS_REGISTRY.supply(0, encoder);
    }

    public static void dumpFrontendTimingMetrics(MetricEncoder encoder) {
        FRONTEND_TIMINGS_METRICS_REGISTRY.supply(0, encoder);
    }

    public static void dumpTraceLogFunctionsMetrics(MetricEncoder encoder) {
        TRACE_LOG_FUNCTIONS_METRICS_REGISTRY.supply(0, encoder);
    }

    public static void dumpBsExportQueueMetrics(MetricEncoder encoder) {
        BS_EXPORT_QUEUE_REGISTRY.supply(0, encoder);
    }

    public static void dumpModExportQueueMetrics(MetricEncoder encoder) {
        MOD_EXPORT_QUEUE_REGISTRY.supply(0, encoder);
    }

    /**
     * Аналогично {@link MetricRegistry#supply(long, MetricConsumer)},
     * но объединяем несколько MetricRegistry в один stream
     * и добавляем к каждой метрике дополнительные метки
     */
    public static void dumpUnifiedAgentClientMetrics(MetricEncoder encoder) {
        encoder.onStreamBegin(-1);
        encoder.onCommonTime(0);
        for (var entry : UNIFIED_AGENT_CLIENT_METRICS_REGISTRIES.entrySet()) {
            String appenderName = entry.getKey();
            MetricRegistry metricRegistry = entry.getValue();

            Labels commonLabels = metricRegistry.getCommonLabels();
            if (!commonLabels.isEmpty()) {
                encoder.onLabelsBegin(commonLabels.size());
                commonLabels.forEach(((MetricConsumer) encoder)::onLabel);
                encoder.onLabelsEnd();
            }
            metricRegistry.append(0, Labels.of("appender", appenderName), encoder);
        }
        encoder.onStreamEnd();
    }

    public static void dumpGrutObjectApiMetrics(MetricEncoder encoder) {
        GRUT_METRICS_REGISTRY.supply(0, encoder);
    }

    /**
     * Регистрация сенсоров, пишущих статистку по Guava-cache
     *
     * @param sensorNamePrefix - префикс названия сенсора
     * @param caches           - набор кэшей
     */
    public static <K, V> void registerGuavaCachesStats(String sensorNamePrefix, Collection<Cache<K, V>> caches) {
        putToRunSensorFunction(sensorNamePrefix, () -> {
            CacheStats stats = null;
            long size = 0;
            for (Cache<?, ?> cache : caches) {
                size += cache.size();
                if (stats == null) {
                    stats = cache.stats();
                } else {
                    stats = stats.plus(cache.stats());
                }
            }

            if (stats != null) {
                GaugeInt64 sizeSensor = SOLOMON_REGISTRY.gaugeInt64(sensorNamePrefix + "_size");
                sizeSensor.set(size);

                GaugeInt64 hitsSensor = SOLOMON_REGISTRY.gaugeInt64(sensorNamePrefix + "_hits");
                hitsSensor.set(stats.hitCount());

                GaugeInt64 missesSensor = SOLOMON_REGISTRY.gaugeInt64(sensorNamePrefix + "_misses");
                missesSensor.set(stats.missCount());

                GaugeInt64 loadsSensor = SOLOMON_REGISTRY.gaugeInt64(sensorNamePrefix + "_loads");
                loadsSensor.set(stats.loadCount());

                GaugeInt64 evictionsSensor = SOLOMON_REGISTRY.gaugeInt64(sensorNamePrefix + "_evictions");
                evictionsSensor.set(stats.evictionCount());

                GaugeInt64 exceptionsSensor = SOLOMON_REGISTRY.gaugeInt64(sensorNamePrefix + "_load_exceptions");
                exceptionsSensor.set(stats.loadExceptionCount());
            }
        });
    }

    /**
     * Регистрация сенсоров, пишущих информацию из числовых аттрибутов jmx-бина
     *
     * @param sensorNamePrefix - перфикс названия сенсора
     * @param commonLabels     - общие метки сенсоров
     * @param beanName         - имя бина
     * @param beanAttrs        - маппинг - имя jmx-атрибута -> имя метрики
     */
    public static void registerJmxAttributes(String sensorNamePrefix, Labels commonLabels, String beanName,
                                             Map<String, String> beanAttrs) {
        ObjectName objName;
        try {
            objName = new ObjectName(beanName);
        } catch (MalformedObjectNameException e) {
            throw new IllegalArgumentException("Incorrect mbean name", e);
        }
        registerJmxAttributes(sensorNamePrefix, commonLabels, objName, beanAttrs);
    }

    /**
     * Регистрация сенсоров, пишущих информацию из числовых аттрибутов jmx-бина
     *
     * @param sensorNamePrefix - префикс названия сенсора
     * @param commonLabels     - общие метки сенсоров
     * @param objName          - имя бина
     * @param beanAttrs        - маппинг - имя jmx-атрибута -> имя метрики
     */
    public static void registerJmxAttributes(String sensorNamePrefix, Labels commonLabels, ObjectName objName,
                                             Map<String, String> beanAttrs) {
        putToRunSensorFunction(objName.toString(), () -> {
            if (!BEAN_SERVER.isRegistered(objName)) {
                return;
            }

            try {
                AttributeList attrsList = BEAN_SERVER.getAttributes(
                        objName,
                        beanAttrs.keySet().toArray(new String[beanAttrs.size()])
                );

                attrsList.asList().forEach(attr -> {
                    GaugeDouble beanAttrSensor = SOLOMON_REGISTRY.gaugeDouble(
                            sensorNamePrefix + "_" + beanAttrs.get(attr.getName()), commonLabels);

                    beanAttrSensor.set(((Number) attr.getValue()).doubleValue());
                });
            } catch (ReflectionException | InstanceNotFoundException e) {
                throw new IllegalStateException("Can't get bean metrics for " + commonLabels.toString(), e);
            }
        });
    }

    /**
     * Найти JMX-бины именами, подходящими под namePattern
     */
    public static Collection<ObjectName> findJmxBeans(String namePattern) {
        MBeanServer beanServer = ManagementFactory.getPlatformMBeanServer();
        try {
            return beanServer.queryNames(new ObjectName(namePattern), null);
        } catch (MalformedObjectNameException e) {
            throw new IllegalArgumentException("Malformed bean pattern", e);
        }
    }

    public static String getCurrentEnv() {
        return Environment.getCached().name().toLowerCase();
    }

    public static void incrementIfNotNull(@Nullable Rate metric) {
        if (metric == null) {
            return;
        }
        metric.inc();
    }

    private static Labels getCommonLabels() {
        return Labels.of(ENV_LABEL_NAME, getCurrentEnv());
    }

    private static Labels getCommonPushLabels() {
        return getCommonLabels()
                .add(HOST_LABEL_NAME, SystemUtils.hostname());
    }

    public static HistogramCollector explicitHistogram(int[] bins) {
        double[] bounds = IntStream.of(bins).asDoubleStream().toArray();
        return Histograms.explicit(bounds);
    }
}
