package ru.yandex.direct.sql.normalizer;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.List;
import java.util.Stack;

/**
 * Класс для сортировки полей в запросе по алфавиту.
 * Сортирует все поля во всех запросах внутри select от select до from или конца запроса
 * и внутри update от set до where, from или конца запроса, включая и учитывая все вложенные select'ы и update'ы.
 * Исходный запрос должен быть предварительно обработан методом {@link QueryParser#parseQuery(String, char[])}.
 *
 * Пример использования:
 *
 * <pre>
 *         String query = ... //тут код получения тела запроса
 *         // Буфер задаем больше в два раза, так как в результате парсинга длина запроса может увеличиться
 *         // за счет добавления разделительных пробелов.
 *         char[] input = new char[query.length() * 2];
 *         int length = QueryParser.parseQuery(query, query.length(), input);
 *         QueryFieldsSorter sorter = new QueryFieldsSorter(buffer, length);
 *         IQueryReader reader = sorter.sortQueryFields();
 *         char[] output = new char[reader.getLength()];
 *         reader.copyToBuffer(output);
 *         String queryWithSortedFields = new String(output);
 * </pre>
 *
 * В результате запрос, например:
 *
 * <pre>
 * select id, name, family from users where family = ? and (group_id = ? or manager_id in (...)) and salary > ?
 * </pre>
 *
 * Будет преобразован в следующую форму:
 *
 * <pre>
 * select family, id, name from users where family = ? and (group_id = ? or manager_id in (...)) and salary > ?
 * </pre>
 *
 * А запрос, например:
 *
 * <pre>
 * update table1 as dest, (select name, id, family from table2 where id = ?) as src set dest.name = src.name, dest.id = src.id, dest.family = src.family where dest.id = ?;
 * </pre>
 *
 * Будет преобразован в:
 *
 * <pre>
 * update table1 as dest, (select family, id, name from table2 where id = ?) as src set dest.family = src.family, dest.id = src.id, dest.name = src.name where dest.id = ?;
 * </pre>
 */
public class QueryFieldsSorter {
    private static final char[] selectChars = "select ".toCharArray();
    private static final char[] fromChars = " from ".toCharArray();
    private static final char[] distinctChars = "distinct ".toCharArray();
    private static final char[] straightJoinChars = "straight_join ".toCharArray();
    private static final char[] setChars = " set ".toCharArray();
    private static final char[] whereChars = " where ".toCharArray();

    // Минимальный размер моля, чтобы искать там селекты и апдейты
    private static final int MIN_FIELD_REMAIN_LENGTH = setChars.length + 3;

    private final char[] buffer;
    private final int len;
    private final int maxFieldsCount;

    private QueryField sortedMainField;
    private int currentFieldsCount = 0;

    /**
     * Создает экземпляр класса для сортировки запроса
     * @param buffer буфер с телом запроса
     * @param len длина запроса в буфере (она может быть меньше, чем размер буфера)
     */
    public QueryFieldsSorter(char[] buffer, int len) {
        this.buffer = buffer;
        this.len = len;
        // Поле в селекте - это минимум буква, запятая и пробел, то есть минимум 3 символа (пробел после запятых
        // обязательно появится после предобработки исходного запроса парсером QueryParser).
        // То есть полей в запросе может быть максимум len / 3 + еще одно поле для корня дерева.
        this.maxFieldsCount = len / 3 + 1;
    }

    private void checkFieldsCount() {
        currentFieldsCount++;
        if (currentFieldsCount > maxFieldsCount) {
            throw new IllegalStateException(String.format("Too many fields: %d, query length: %d, query:%n%s",
                    maxFieldsCount, len, new String(buffer, 0, len)));
        }
    }

    /**
     * Рекурсивно строит дерево полей в запросе и возвращает корень
     * @return корень дерева полей в запросе
     */
    private QueryField makeOuterField() {
        int pos = 0;
        QueryField field = new QueryField(0);
        while (pos < len - MIN_FIELD_REMAIN_LENGTH) {
            // Пока имеет смысл искать, проверяем, не является ли текущая позиция началом селекта или блока set апдейта
            int beginGroup = tryToFindSortableFieldsGroupBegin(pos);
            if (beginGroup > 0) {
                // Если является, то добавляем группу полей из селекта или блока set апдейта
                QuerySortableFieldsGroup sortableFieldsGroup = findSortableFieldsGroup(beginGroup);
                field.sortableGroups.add(sortableFieldsGroup);
                checkFieldsCount();
                pos = sortableFieldsGroup.fields.get(sortableFieldsGroup.fields.size() - 1).end + fromChars.length;
            }
            pos++;
        }
        field.end = len;
        return field;
    }

    private boolean isSelectBeginAtPos(int pos) {
        return (pos == 0 || (pos > 0 && (buffer[pos - 1] == ' ' || buffer[pos - 1] == '('))) &&
                pos + selectChars.length <= len &&
                Arrays.equals(
                        selectChars, 0, selectChars.length,
                        buffer, pos, pos + selectChars.length);
    }

    private boolean isArrayBeginAtPos(int pos, char[] chars) {
        return (pos + chars.length <= len &&
                Arrays.equals(chars, 0, chars.length, buffer, pos, pos + chars.length));
    }

    /**
     * Проверяет, не начинается ли с позиции pos в массиве {@link #buffer} селект или блок set апдейта
     * @param pos позиция в массиве {@link #buffer}
     * @return true, если с позиции pos в массиве {@link #buffer} начинается селект или блок set апдейта, иначе false
     */
    private int tryToFindSortableFieldsGroupBegin(int pos) {
        int beginSelect = -1;
        if (isSelectBeginAtPos(pos)) {
            beginSelect = correctSelectBegin(pos);
        } else if (isArrayBeginAtPos(pos, setChars)) {
            beginSelect = pos + setChars.length;
        }
        return beginSelect;
    }

    /**
     * Проверяет, не идет ли сразу после объявления селекта distict или straight_join (или оба сразу), и если да, то
     * корректирует позицию, откуда искать поля селекта
     * @param beginSelect позиция объявления select в масииве {@link #buffer}
     * @return скорректированная позиция, откуда искать поля
     */
    private int correctSelectBegin(int beginSelect) {
        beginSelect += selectChars.length;
        while (true) {
            if (beginSelect + distinctChars.length < len && Arrays.equals(
                    distinctChars, 0, distinctChars.length,
                    buffer, beginSelect, beginSelect + distinctChars.length)) {
                beginSelect += distinctChars.length;
            } else if (beginSelect + straightJoinChars.length < len && Arrays.equals(
                    straightJoinChars, 0, straightJoinChars.length,
                    buffer, beginSelect, beginSelect + straightJoinChars.length)) {
                beginSelect += straightJoinChars.length;
            } else {
                break;
            }
        }
        return beginSelect;
    }

    /**
     * Собирает группу полей select или блока set апдейта
     * @param beginGroup позиция в массиве {@link #buffer}, откуда искать поля
     * @return группа полей select или блока set апдейта
     */
    private QuerySortableFieldsGroup findSortableFieldsGroup(int beginGroup) {
        int pos = beginGroup;
        QuerySortableFieldsGroup sortableFieldsGroup = new QuerySortableFieldsGroup(beginGroup);
        QueryField field = new QueryField(beginGroup);
        sortableFieldsGroup.fields.add(field);
        checkFieldsCount();
        while (pos < len) {
            // Пока есть возможность, ищем очередное поле
            if (makeInnerField(field) || field.end + 2 >= len) {
                // поля закончились
                break;
            }
            // так как все поля разделены запятой с пробелом ", ", то новое поле ищем через два символа
            // после окончания текущего
            field = new QueryField(field.end + 2);
            sortableFieldsGroup.fields.add(field);
            checkFieldsCount();
            pos = field.begin;
        }
        sortableFieldsGroup.end = field.end;
        return sortableFieldsGroup;
    }

    /**
     * Формирует внутреннее поле внутри селекта или блока set апдейта, с учетом того, что каждое моле может содержать
     * внутренние блоки селектов и апдейтов
     * @param field поле, которое нужно сформировать
     * @return true, если найденное поле последнее в блоке и дальше искать в этом блоке нет смысла
     */
    private boolean makeInnerField(QueryField field) {
        int pos = field.begin;
        int bracesCount = 0;
        while (pos <= len) {
            if (pos == len || (bracesCount == 0 &&
                    (buffer[pos] == ';' || isArrayBeginAtPos(pos, fromChars) || isArrayBeginAtPos(pos, whereChars)))) {
                // Если доискали до конца запроса или встретили where, или from, то завершаем поле, с гарантией того,
                // что оно в данном блоке последнее
                break;
            }
            // Проверяем, нет ли в этом поле других вложенных блоков
            int beginGroup = tryToFindSortableFieldsGroupBegin(pos);
            if (beginGroup > 0) {
                // Если есть, формируем новый блок внутри поля
                QuerySortableFieldsGroup sortableFieldsGroup = findSortableFieldsGroup(beginGroup);
                field.sortableGroups.add(sortableFieldsGroup);
                checkFieldsCount();
                pos = sortableFieldsGroup.fields.get(sortableFieldsGroup.fields.size() - 1).end + fromChars.length;
            }
            // Подсчитываем скобочки, что бы найти именно ту запятую, которая разделяет поля этого блока, а не
            // что-нибудь еще.
            char c = buffer[pos];
            if (c == '(') {
                bracesCount++;
            }
            if (c == ')') {
                bracesCount--;
            }
            if (c == ',' && bracesCount == 0) {
                // Нашли ту самую запятую, но возвращаем, что в этом блоке еще точно есть поля.
                field.end = pos;
                return false;
            }
            pos++;
        }
        // Полей в блоке точно больше нет
        field.end = pos;
        return true;
    }

    /**
     * Рекурсивно сортирует поле field
     * @param field поле для сортировки
     */
    private void sortField(QueryField field) {
        for (QuerySortableFieldsGroup group: field.sortableGroups) {
            // Просто сортируем все группы полей внутри этого поля
            sortSortableFieldsGroup(group);
        }
    }

    /**
     * Сортирует группу полей
     * @param group группа полей для сортировки
     */
    private void sortSortableFieldsGroup(QuerySortableFieldsGroup group) {
        for (QueryField field: group.fields) {
            // Сначала сортируем каждое поля внутри группы отдельно. Вдруг там есть вложенные?
            sortField(field);
        }
        // Теперь можем их сортировать по алфавиту. При этом реально сортируются только координаты полей,
        // а весь текст остается как есть
        group.fields.sort(createFieldsComparator());
    }

    /**
     * Сначала сортирует все поля во всех запросах внутри select от select до from или конца запроса,
     * и внутри update от set до where, from или конца запроса, включая вложенные select'ы и update'ы.
     * Затем возвращает читателя финального запроса с корректно отсортированными полями
     * @return Читатель финального запроса с корректно отсортированными полями
     */
    public IQueryReader sortQueryFields() {
        if (sortedMainField == null) {
            currentFieldsCount = 0;
            sortedMainField = makeOuterField();
            sortField(sortedMainField);
        }
        return getReader(sortedMainField);
    }

    /**
     * Возвращает читателя финального запроса с корректно отсортированными полями для поля field
     * @param field поле для чтения
     * @return читатель финального запроса с корректно отсортированными полями для поля field
     */
    private IQueryReader getReader(QueryField field) {
        return new QueryFieldReader(field, buffer);
    }

    /**
     * Создает компаратора для сравнения двух полей запроса
     * @return компаратор для сравнения двух полей запроса
     */
    private Comparator<QueryField> createFieldsComparator() {
        return (qf1, qf2) -> {
            // Если поля без вложенных групп, то можем просто посимвольно сравнить их содержимое
            if (qf1.sortableGroups.size() == 0 && qf2.sortableGroups.size() == 0) {
                return Arrays.compare(buffer, qf1.begin, qf1.end, buffer, qf2.begin, qf2.end);
            }
            // Если хотя бы одно поле содержит группы, то создаем читателей этих полей,
            // и посимвольно сравниваем их содержимое
            int len1 = Math.min(qf1.length(), qf2.length());
            IQueryReader r1 = getReader(qf1);
            IQueryReader r2 = getReader(qf2);
            for (int i = 0; i < len1; i++) {
                int compare = r1.getNextChar() - r2.getNextChar();
                if (compare != 0) {
                    return compare;
                }
            }
            return r1.getLength() - r2.getLength();
        };
    }

    /**
     * Поле, с вложенными группами полей для сортировки
     */
    private static class QueryField {
        private final int begin;
        private int end;
        private final List<QuerySortableFieldsGroup> sortableGroups = new ArrayList<>();

        private QueryField(int begin) {
            this.begin = begin;
        }

        private int length() {
            return end - begin;
        }
    }

    /**
     * Группа полей для сортировки
     */
    private static class QuerySortableFieldsGroup {
        private final int begin;
        private int end;
        private final List<QueryField> fields = new ArrayList<>();

        private QuerySortableFieldsGroup(int begin) {
            this.begin = begin;
        }
    }

    /**
     * Рекурсивный читатель полей запроса
     */
    private static class QueryFieldReader implements IQueryReader {
        private final int length;
        private final Stack<Object> stack = new Stack<>();
        private FieldWithCounter currentFieldWithCounter;
        private final char[] buffer;
        private int pos;

        public QueryFieldReader(QueryField field, char[] buffer) {
            length = field.end - field.begin;
            currentFieldWithCounter = new FieldWithCounter(field, false);
            this.pos = 0;
            this.buffer = buffer;
        }

        @Override
        public int getLength() {
            return length;
        }

        @Override
        public boolean hasNextChar() {
            return pos < length;
        }

        /**
         * Выдает следующий символ поля
         * @return следующий символ поля
         */
        @Override
        public char getNextChar() {
            while (currentFieldWithCounter.readToEnd() || currentFieldWithCounter.readToLocalEnd()) {
                if (currentFieldWithCounter.readToEnd()) {
                    // Если текущее поле прочитано до конца, то берем из стека его группу
                    SortableFieldGroupWithCounter groupWithCounter = (SortableFieldGroupWithCounter) stack.pop();
                    if (groupWithCounter.readToEnd()) {
                        // Если группа тоже прочитана до конца, то берем из стека поле, внутри которого была эта группа,
                        // это поле будет проверено в следующей итерации while
                        currentFieldWithCounter = (FieldWithCounter) stack.pop();
                    } else {
                        // А если группа еще не прочитана до конца, то берем следующее поле в ней
                        currentFieldWithCounter = groupWithCounter.getNextField();
                        // И кладем группу назад в стек
                        stack.push(groupWithCounter);
                    }
                } else {
                    // Если текущее поле прочитано до локального конца, то есть в нем еще остались
                    // сортированные группы, но все символы до очередной группы прочитаны, надо найти в нем первое
                    // поле следующей группы. Для этого кладем в стек текущее поле
                    stack.push(currentFieldWithCounter);
                    // Берем следующую группу в поле
                    QuerySortableFieldsGroup group = currentFieldWithCounter.nextGroup();
                    // Формируем для нее счетчики и кладем группу в стек
                    SortableFieldGroupWithCounter groupWithCounter = new SortableFieldGroupWithCounter(group);
                    stack.push(groupWithCounter);
                    // Берем очередное поле группы
                    currentFieldWithCounter = groupWithCounter.getNextField();
                }
            }
            pos++;
            // Возвращаем очередной символ текущего поля
            return currentFieldWithCounter.nextChar();
        }

        /**
         * Итератор полей в группе
         */
        private class SortableFieldGroupWithCounter {
            private final List<FieldWithCounter> fieldsWithCounters;
            private int childIndex = 0;

            public SortableFieldGroupWithCounter(QuerySortableFieldsGroup group) {
                fieldsWithCounters = new ArrayList<>();
                for (int i = 0; i < group.fields.size(); i++) {
                    QueryField field = group.fields.get(i);
                    boolean notLast = i != group.fields.size() - 1;
                    fieldsWithCounters.add(new FieldWithCounter(field, notLast));
                }
            }

            public boolean readToEnd() {
                return childIndex == fieldsWithCounters.size();
            }

            public FieldWithCounter getNextField() {
                return fieldsWithCounters.get(childIndex++);
            }
        }

        /**
         * Итератор групп полей, а также символов перед, между, и после групп полей
         */
        private class FieldWithCounter {
            private final QueryField field;
            private final int addition;

            private int childIndex = 0;
            private int localPos = 0;
            private int globalBegin;
            private int localEnd;

            /**
             * Создает поле с индексами прочитанных групп и позициями символов внутри участка до, между и после групп
             * @param field поле
             * @param notLastInGroup не является ли поле последним в группе полей
             */
            public FieldWithCounter(QueryField field, boolean notLastInGroup) {
                // Если поле не последнее, то после того, как все его символы будут прочитаны, нужно в конце добавить
                // запятую и пробел
                addition = notLastInGroup ? 2 : 0;
                this.field = field;
                // Текущий блок символов группы - либо до начала первой группы, либо до конца, если групп нет
                localEnd = field.sortableGroups.size() == 0 ?
                        field.length() : field.sortableGroups.get(0).begin - field.begin;
                globalBegin = field.begin;
            }

            public boolean readToEnd() {
                // Проверяет, завершилось ли чтение всего поля с учетом добавочного разделителя полей
                // из запятой с пробелом, который нужно отдать, если все символы группы прочитаны,
                // а поле не последнее в своей группе
                return childIndex == field.sortableGroups.size() && localPos == localEnd + addition;
            }

            public boolean readToLocalEnd() {
                // Проверяет, завершилось ли чтение текущего участка между группами полей (или участка
                // до первой или после последней группы), с учетом добавочного разделителя полей из запятой с пробелом,
                // который нужно отдать, если все символы группы прочитаны, а поле не последнее в своей группе
                return childIndex == field.sortableGroups.size() ?
                        localPos == localEnd + addition : localPos == localEnd;
            }

            public QuerySortableFieldsGroup nextGroup() {
                // Переключаемся на следующий промежуток между группами полей и возвращаем текущую группу
                QuerySortableFieldsGroup group = field.sortableGroups.get(childIndex++);
                globalBegin = group.end;
                localPos = 0;
                // Считаем длину очередного участка, он от конца текущей группы до начала следующий, если она есть,
                // либо до конца поля, если групп больше нет
                localEnd = childIndex == field.sortableGroups.size() ?
                        field.end - globalBegin : field.sortableGroups.get(childIndex).begin - globalBegin;
                return group;
            }

            public char nextChar() {
                char c;
                if (localPos < localEnd) {
                    // Если читаем в пределах участка, то просто возвращаем очередной символ
                    c = buffer[globalBegin + localPos];
                } else if (localPos == localEnd) {
                    // если просят прочитать символ сразу после окончания локального участка, значит началось чтение
                    // разделителя между полями. Отдаем начала запятую
                    c = ',';
                } else {
                    // а потом пробел. Больше двух символов после конца участка читать не должны, сработает условие
                    // readToLocalEnd() или readToEnd()
                    c = ' ';
                }
                localPos++;
                return c;
            }
        }
    }

}
