package ru.yandex.direct.sql.normalizer;

/**
 * Класс для создания нормализованных форм запроса. Рекомендуется использовать один экземпляр на поток, для нормализации
 * множества разных запросов. Внутри себя содержит выделенные буферы, чтобы на каждую нормализацию не аллоцировать их
 * заново.
 */
public class QueryNormalizer {
    private final char[] defaultQueryInputBuffer;
    private final char[] defaultQueryOutputBuffer;

    private final int maxClosingBracesInNormalForm;
    private final int maxNormalFormLength;
    private final int bufferSize;

    /**
     * Создает нормализатор запросов с указанными максимальной допустимой длиной нормальной формы,
     * максимально допустимым числом закрывающих круглых скобок и буферами заданного размера.
     * @param maxClosingBracesInNormalForm максимально допустимое число закрывающих круглых скобок. Будет
     *                                     проигнорировано при нормализации, если нормализованный запрос получится
     *                                     меньше максимальной допустимой длины нормальной формы.
     * @param maxNormalFormLength максимально допустимая длина нормальной формы запроса
     * @param bufferSize размер буферов. Для эффективной работы нужно задавать размер буферов в два раза больше
     *                   максимальной длины исходных нормализуемых запросов
     */
    public QueryNormalizer(int maxClosingBracesInNormalForm, int maxNormalFormLength, int bufferSize) {
        this.maxClosingBracesInNormalForm = maxClosingBracesInNormalForm;
        this.maxNormalFormLength = maxNormalFormLength - 4;
        this.bufferSize = bufferSize;
        defaultQueryInputBuffer = new char[bufferSize];
        defaultQueryOutputBuffer = new char[bufferSize];
    }

    /**
     * Возвращает нормализованную форму запроса. Запрос сначала нормализуется
     * методом {@link QueryParser#parseQuery(String, char[])}, затем, если нормальная форма запроса
     * получилась слишком большой, то она обрезается до максимально допустимой длины, или до последней закрывающей
     * круглой скобки, и в конце в получившейся нормальной форме сортируются поля селектов и блоков set апдейтов
     * классом {@link QueryFieldsSorter}
     * @param query исходный запрос
     * @return нормализованная форма запроса
     */
    public String normalizeQuery(String query) {
        char[] queryInputBuffer = this.defaultQueryInputBuffer;
        char[] queryOutputBuffer = this.defaultQueryOutputBuffer;

        if (query.length() * 2 > bufferSize) {
            queryInputBuffer = new char[bufferSize];
            queryOutputBuffer = new char[bufferSize];
        }

        int length = QueryParser.parseQuery(query, queryInputBuffer);

        if (length > maxNormalFormLength + 4) {
            length = trimToMaxLength(queryInputBuffer, length, maxClosingBracesInNormalForm, maxNormalFormLength);
        }

        QueryFieldsSorter sorter = new QueryFieldsSorter(queryInputBuffer, length);
        IQueryReader reader = sorter.sortQueryFields();
        reader.copyToBuffer(queryOutputBuffer);
        return new String(queryOutputBuffer, 0, reader.getLength());
    }

    /**
     * Обрезает запрос до maxNormalFormLength символов или до maxClosingBracesInNormalForm закрывающих круглых скобок,
     * в зависимости от того, что наступит раньше. Если максимально допустимая длина запроса наступит раньше,
     * но от последней найденной, в пределах разрешенной длины символов, закрывающей скобки до максимально допустимой
     * длины останется менее 10% от допустимой длины, то запрос обрежется по последней найденной скобки.
     * Это позволяет эффективно группировать запросы с огромными перечислениями удаляемых или добавляемых строк в конце.
     * Если запрос обрезан, добавляет к нему в конце пробел (если обрезалось не на пробеле) и три точки '...'
     * @param buffer буфер с символами запроса
     * @param bufferLength длина запроса в буфере
     * @param maxClosingBracesInNormalForm максимальное количество закрывающих круглых скобок, после которых запрос
     *                                     будет обрезан
     * @param maxNormalFormLength максимально допустимая длина запроса
     * @return длинна обрезанного запроса (или bufferLength, если bufferLength < maxNormalFormLength)
     */
    private static int trimToMaxLength(
            char[] buffer, int bufferLength,
            int maxClosingBracesInNormalForm, int maxNormalFormLength) {
        if (bufferLength < maxNormalFormLength) {
            return bufferLength;
        }
        int pos = 0;
        int foundCount = 0;
        int lastPos = 0;
        while (foundCount < maxClosingBracesInNormalForm && pos < maxNormalFormLength) {
            char c = buffer[pos];
            if (c == ')') {
                foundCount++;
                lastPos = pos;
            }
            pos++;
        }
        lastPos++;
        int length;
        if (foundCount == maxClosingBracesInNormalForm || pos - lastPos < maxNormalFormLength / 10) {
            length = lastPos;
        } else {
            length = pos;
        }
        if (length > 0 && buffer[length - 1] != ' ') {
            // Если обрезали не на пробеле, то добавим его, чтобы не было проблем с повторной нормализацией
            // уже нормализованного запроса
            buffer[length++] = ' ';
        }
        for (int i = 0; i < 3; i++) {
            buffer[length++] = '.';
        }
        return length;
    }

}
