package ru.yandex.direct.sql.normalizer;

/**
 * Заменяет в запросе все значения параметров на символы знака вопроса, а так же приводит пробельные отступы
 * между "токенами" запроса к единому стандарту.
 */
public class QueryParser {
    // Битовая маска "символов-разделителей" для ascii-кодов 0-64
    private static long delimiters1 = 0;
    // Битовая маска символов-разделителей для ascii-кодов 64-128
    private static long delimiters2 = 0;

    // Заполняем битовые маски "символов-разделителей"
    static {
        for (int i = 0; i < 64; i++) {
            char c = (char) i;
            if (c == ';' || c == '<' || c == ',' || c == '>' || c == '=' || c == '-' || c == '+'
                    || c == '*' || c == '%' || c == '/' || c == '!') {
                delimiters1 |= (1L << i);
            }
        }
        for (int i = 64; i < 128; i++) {
            char c = (char) i;
            if (c == '^' || c == '~' || c == '|') {
                delimiters2 |= (1L << (i - 64));
            }
        }
    }

    /***
     * Закрываем конструктор, так как можно пользоваться только статическими методами класса
     */
    private QueryParser() {}

    /**
     * Парсит первые queryLength символов из запроса query и записывает результат в массив buffer. Возвращает количество
     * записанных символов в массив buffer. Во время парсинга делает следующие вещи:
     * <ol>
     *     <li>Переводит все символы запроса в нижний регистр;</li>
     *     <li>Заменяет все параметры запроса на символ знака вопроса '?' (без кавычек);</li>
     *     <li>Нормализует пробельные символы (whitespaces) по следующим правилам:
     *         <ul>
     *             <li>Все пробельные символы схлопываются в один пробел ' ', однако после
     *             открывающей круглой скобки '(' и перед запятой ',', точкой с запятой ';' или
     *             закрывающей круглой скобкой ')' удаляются даже одиночные пробелы. Так же удаляются все пробельные
     *             символы с конца и начала запроса (аналогично {@link String#trim()});
     *             </li>
     *             <li>Все группы подряд идущих "символов-разделителей"
     *             ('+', '-', '*' '/', '%', '!', '^', '~', '=', '&gt;', '&lt;', ',', '|'), если они не являются
     *             частью числа, и не возникает противоречий с предыдущим пунктом, отбиваются пробелами с двух сторон,
     *             даже если в исходном запросе никаких пробельных символов там не было. Например,
     *             <nowrap>a&gt;=b&nbsp;+c</nowrap> превратится в <nowrap>a &gt;= b + c</nowrap>;
     *             </li>
     *         </ul>
     *     </li>
     *     <li>Удаляет все многострочные комментарии в блоках &#47;&#42; &#42;&#47; и однострочные
     *     от -- до конца строки как будто их вообще там не было. То есть комментарии не учитываются вообще,
     *     и не заменяются на пробел;</li>
     *     <li>Заменяет всё, что находится внутри скобок секции <code>in (param1, param2 ... paramN)</code>
     *     на три точки '...' без кавычек;</li>
     * </ol>
     * Например, запрос:
     *
     * <pre>
     * SELECT &#47;&#42; method:getPetrovs &#42;&#47;
     * id , name,family
     * FROM users
     * WHERE
     *     family="Petrov" and -- get all Petrov's
     *     (
     *         group_id =-1 or manager_id in ( '1', '3', '8' , '92' )
     *     )
     *     and salary>-20e-15;
     * </pre>
     *
     * Будет преобразован в следующую одиночную строку:
     *
     * <pre>
     * select id, name, family from users where family = ? and (group_id = ? or manager_id in (...)) and salary > ?
     * </pre>
     *
     * Метод реализован в виде конечного автомата и работает за один проход по исходной строке.
     * @param query исходный sql-запрос
     * @param buffer - буфер, куда записывать результат. Так как в результате парсинга, итоговая длина запроса может
     *               превысить исходную длину запроса queryLength (за счет добавленных пробелов), рекомендуется
     *               передавать буфер размером в 2 * queryLength
     * @return длина распарсеного запроса в буфере buffer.
     */
    public static int parseQuery(String query, char[] buffer) {
        // Флаг того, что мы внутри кавычек одного из параметров внутри секции in ()
        boolean isInInQuota = false;
        // Флаг того, что уже встретили ' in', но еще не встретили открывающую скобку '('
        boolean maybeIn = false;
        // Последний непробельный прочитанный символ запроса
        char lastNotWhitespaceChar = '(';
        // Символ текущих кавычек. Может быть '\'' или '"'
        char quotaChar = 0;
        // Позиция начала очередного записанного токена в массиве buffer;
        int beginTokenBufferPos = 0;
        // Позиция в исходном запросе
        int pos = 0;
        // Позиция в результате
        int bufferPos = 0;
        // Сколько открытых скобок внутри секции in (   ), включая самую первую.
        int inInBracesCount = 0;
        // Какой тип "токена" из запроса мы сейчас обрабатываем
        TokenStateEnum tokenState = TokenStateEnum.IN_WHITESPACES;
        // Состояние обработки запроса
        ParseStateEnum parseState = ParseStateEnum.IN_TOKEN;
        int queryLength = query.length();
        while (pos < queryLength) {
            char c = query.charAt(pos);
            char nextChar = pos == queryLength - 1 ? ' ' : query.charAt(pos + 1);
            // Смотрим, в каком состоянии мы находимся, и стоит ли переходить в другое
            switch (parseState) {
                case IN_MULTILINE_COMMENT:
                    // Мы внутри мультистрочного комментария, который нужно просто пролистать
                    if (c == '/' && query.charAt(pos - 1) == '*') {
                        // Если он закончился - переходим к обработке значимых символов запроса.
                        parseState = ParseStateEnum.IN_TOKEN;
                    }
                    break;
                case IN_SINGLE_LINE_COMMENT:
                    // Мы внутри однострочного комментария, который нужно просто пролистать
                    if (c == '\n') {
                        // Если он закончился - переходим к обработке значимых символов запроса.
                        parseState = ParseStateEnum.IN_TOKEN;
                    }
                    break;
                case IN_QUOTA:
                    // Мы внутри кавычек, которые надо пролистать и заменить весь блок на символ знака вопроса
                    if (c == quotaChar) {
                        // Мы опять встретили символ кавычек, возможно он закрывающий
                        if (nextChar == c) {
                            // Но символ оказался экранирован повтором той же кавычки, пропускаем его,
                            // и экранированный тоже
                            pos++;
                        } else {
                            // Если кавычки не экранированы, то это конец блока c кавычками. Пишем в буфер знак вопроса,
                            // и переходим в состояние обработки токена в стадии обработки пробельных символов,
                            // чтобы знак вопроса не слипся со словом, если оно идет дальше впритык
                            buffer[bufferPos++] = '?';
                            lastNotWhitespaceChar = '?';
                            beginTokenBufferPos = bufferPos + 1;
                            tokenState = TokenStateEnum.IN_WHITESPACES;
                            parseState = ParseStateEnum.IN_TOKEN;
                        }
                    } else if (c == '\\') {
                        // если кавычек не встретили, но встретили обратный слэш,
                        // то пропускаем его и следующий за ним символ
                        pos++;
                    }
                    break;
                case IN_IN:
                    // Мы внутри секции in (...)
                    if (isInInQuota) {
                        // Мы внутри кавычек какого-то параметра из секции in (...), который нужно просто пролистать
                        // В этом состоянии мы не учитываем закрывающие скобки, так эти скобки - значение параметра,
                        // а не закрывающие скобки in
                        if (c == quotaChar) {
                            // Мы опять встретили символ кавычек, возможно он закрывающий
                            if (nextChar == c) {
                                // Но символ оказался экранирован повтором той же кавычки, пропускаем его,
                                // и экранированный тоже
                                pos++;
                            } else {
                                // Если кавычки не экранированы, то это конец блока с кавычками.
                                // Выходим из состояния пролистывания параметров в кавычках,
                                // и снова начинаем учитывать закрывающие скобки.
                                isInInQuota = false;
                            }
                        } else if (c == '\\') {
                            // Если кавычек не встретили, но встретили обратный слэш,
                            // то пропускаем его и следующий за ним символ
                            pos++;
                        }
                    } else {
                        // Мы вне кавычек параметров из секции in (...)
                        if (c == '\'' || c == '"') {
                            // Если встретили кавычку, запоминаем какую, чтобы правильно найти закрывающую
                            // и переходим в состояние пролистывания значения параметра в кавычках внутри блока in (...)
                            quotaChar = c;
                            isInInQuota = true;
                        } else if (c == '(') {
                            // Встретили открывающую скобку, нужно увеличить счетчик открытых скобок
                            inInBracesCount++;
                        } else if (c == ')') {
                            // А если встретили закрывающую, то уменьшить счетчик открытых скобок
                            inInBracesCount--;
                            if (inInBracesCount == 0) {
                                // Если счетчик скобок равен нулю, это конец блока in (...). Записываем его в буфер,
                                // и переходим к стадии обработки значимых символов запроса
                                bufferPos = appendString(buffer, " in (...)", bufferPos);
                                beginTokenBufferPos = bufferPos + 1;
                                lastNotWhitespaceChar = ')';
                                tokenState = TokenStateEnum.IN_WHITESPACES;
                                parseState = ParseStateEnum.IN_TOKEN;
                            }
                        }
                    }
                    break;
                default:
                    // Мы в стадии обработки значимых символов запроса
                    if (c == '\'' || c == '"') {
                        // Если встретили кавычку, запоминаем какую, чтобы правильно найти закрывающую
                        // и переходим в состояние пролистывания значения параметра в кавычках
                        quotaChar = c;
                        parseState = ParseStateEnum.IN_QUOTA;
                        if (tokenState == TokenStateEnum.IN_WORD && bufferPos > beginTokenBufferPos) {
                            // Если до этого мы обрабатывали какое-то слово, то возможно что оно число,
                            // и его нужно заменить на знак вопроса
                            bufferPos = changeIfNumber(buffer, beginTokenBufferPos, bufferPos);
                        }
                        if (lastNotWhitespaceChar != '(') {
                            // Если предыдущий значимый символ не был открывающей скобкой, то добавляем пробел
                            buffer[bufferPos++] = ' ';
                        }
                    } else {
                        // Кавычку мы не встретили
                        if (maybeIn) {
                            // Но до этого встретили ' in' (без кавычек), возможно, это начало блока in (...)
                            if (c == '(' || Character.isWhitespace(c)) {
                                // Если текущий символ - открывающая скобка или пробельный символ,
                                // то значит это точно начало блока in (...),
                                // поэтому переходим в соответствующее состояние
                                parseState = ParseStateEnum.IN_IN;
                                maybeIn = false;
                                // Правильно устанавливаем счетчик скобок
                                inInBracesCount = c == '(' ? 1 : 0;
                            } else {
                                // Если текущий символ не открывающая скобка и не пробельный, значит это точно не начало
                                // блока in (...), а какое обычное слово, начинающееся на in.
                                maybeIn = false;
                                // дописываем ' in' в буфер, и переходим в состояние обработки слова
                                bufferPos = appendString(buffer, " in", bufferPos);
                                tokenState = TokenStateEnum.IN_WORD;
                                beginTokenBufferPos = bufferPos - 2;
                            }
                        } else if ((c == 'i' || c == 'I')
                                && (nextChar == 'n' || nextChar == 'N')
                                && tokenState == TokenStateEnum.IN_WHITESPACES) {
                            // Не встретили кавычку, но если встретили 'in' и предыдущий символ пробельный, то возможно
                            // это начало блока in (...). Проскакиваем n и переходим в состояние обработки возможного
                            // начала блока in (...)
                            pos++;
                            maybeIn = true;
                        }
                        if (parseState != ParseStateEnum.IN_IN && !maybeIn) {
                            // Мы не встретили кавычек, а так же не в состоянии обработки начала,
                            // или самого блока in (...). А вдруг мы в начале комментария?
                            if (c == '/' && nextChar == '*') {
                                // И действительно, начало мультистрочного комментария.
                                // Переходим в состояние его обработки
                                pos += 2;
                                parseState = ParseStateEnum.IN_MULTILINE_COMMENT;
                            } else if (c == '-' && nextChar == '-') {
                                // Мы встретили однострочный комментарий.
                                // Переходим в состояние его обработки
                                pos += 2;
                                parseState = ParseStateEnum.IN_SINGLE_LINE_COMMENT;
                            } else {
                                // Нет, не комментарий и не кавычка и не блок in (...) и не его начало.
                                // Надо проверить, не попали ли мы на экспоненциальную форму числа, например, 23e+34
                                // или даже -2e-7. А мы на нее попали, если обрабатываем слово, текущий символ
                                // '+' или '-', следующий символ слова - цифра, предыдущий символ 'e',
                                // а перед 'е' тоже цифра (привет Кащею).
                                boolean isExponentialForm = (c == '+' || c == '-')
                                        && tokenState == TokenStateEnum.IN_WORD
                                        && bufferPos > beginTokenBufferPos + 1
                                        && buffer[bufferPos - 1] == 'e'
                                        && Character.isDigit(nextChar)
                                        && Character.isDigit(buffer[bufferPos - 2]);

                                if (!isExponentialForm) {
                                    // Нет, это не экспоненциальная форма числа. Узнаем, не обрабатывали ли мы слово
                                    // в этой стадии обработки значимых символов запроса
                                    boolean tokenStageWasNotInWord = tokenState != TokenStateEnum.IN_WORD;

                                    // Вычисляем, к какому состоянию обработки слова относится текущий символ
                                    TokenStateEnum currentTokenPhase = getTokenStateByChar(
                                            c, nextChar, tokenStageWasNotInWord);

                                    if (currentTokenPhase != tokenState) {
                                        // Возможно нужно заменить текущее слово на знак вопроса, если оно число
                                        if (tokenState == TokenStateEnum.IN_WORD && bufferPos > beginTokenBufferPos) {
                                            // Да, мы обрабатывали именно какое-то слово, а не скобки,
                                            // пробельные символы или разделители, поэтому, возможно, что
                                            // это слово - число, и его нужно заменить на знак вопроса
                                            bufferPos = changeIfNumber(buffer, beginTokenBufferPos, bufferPos);
                                        }
                                        // При смене стадии обработки значимых символов запроса нужно добавить пробел
                                        // перед новым блоком, если текущий значимый символ - не пробельный,
                                        // не закрывающая скобка, не запятая или точка с запятой, а предыдущий
                                        // непробельный символ - не открывающая скобка.

                                        // Кроме того, не ставим пробел, если встретили скобку, а до этого
                                        // обрабатывали слово. Это довольно тонкий момент, например, если в запросе
                                        // встретится какая-то функция, но до ее открывающей скобки будет пробел, то он
                                        // останется, а если его не было - то не появится. То есть с одной стороны
                                        // не очень нормализованно выходит, а с другой, если всегда ставить пробел
                                        // в таких случаях, то все открывающие скобки функций будут отделяться пробелом
                                        // от имени функции, а если не ставить никогда, но в запросе встретится,
                                        // например, 'and (...', то пробел после and исчезнет, и будет 'and(...',
                                        // что тоже не очень здорово.
                                        // Разделить эти случаи без построения полноценного дерева запросов
                                        // и определения списка ключевых слов нереально, поэтому было принято
                                        // волевое решение не трогать пробел после возможного имени функции,
                                        // если он есть, и не добавлять его, если его нет. В подавляющем большинстве
                                        // случаев ни ORM, ни программисты не ставят пробелы после имени функции,
                                        // так что принятое решение не навредит.
                                        if (lastNotWhitespaceChar != '(' && c != ',' && c != ')' && c != ';'
                                                && currentTokenPhase != TokenStateEnum.IN_WHITESPACES
                                                && !(currentTokenPhase == TokenStateEnum.IN_BRACES
                                                && tokenState == TokenStateEnum.IN_WORD)) {
                                            buffer[bufferPos++] = ' ';
                                        }
                                        beginTokenBufferPos = bufferPos;
                                        tokenState = currentTokenPhase;
                                    }
                                }
                                if (c != '`' && tokenState != TokenStateEnum.IN_WHITESPACES) {
                                    // Если текущий символ не пробельный, и не '`' (в mySQL он экранирует названия
                                    // полей, но для нормальной формы запроса это не важно), то пишем символ в буфер
                                    lastNotWhitespaceChar = c;
                                    buffer[bufferPos++] = Character.toLowerCase(c);
                                }
                            }
                        }
                    }
            }
            // Переходим к следующему символу
            pos++;
        }
        // Запрос кончился. Надо понять на какой стадии мы остановились, и возможно что-то изменить в буфере
        if (parseState == ParseStateEnum.IN_IN | maybeIn) {
            // Мы были либо в блоке in (...) либо в подозрении на него. Просто допишем ' in (...)' в буфер
            bufferPos = appendString(buffer, " in (...)", bufferPos);
        } else if (parseState == ParseStateEnum.IN_QUOTA) {
            // Мы были в стадии пролистывания значения параметра в кавычках. Добавим знак вопроса в буфер
            buffer[bufferPos++] = '?';
        } else if (tokenState == TokenStateEnum.IN_WORD && bufferPos > beginTokenBufferPos) {
            // Мы обрабатывали слово из запроса, возможно что оно - число, и нужно заменить его на знак вопроса
            bufferPos = changeIfNumber(buffer, beginTokenBufferPos, bufferPos);
        }
        // Возможно мы добавили пробел в конец запроса на предыдущих стадиях обработки, надо его убрать
        while (bufferPos > 1 && (Character.isWhitespace(buffer[bufferPos - 1]) || buffer[bufferPos - 1] == ';')) {
            bufferPos--;
        }
        // Возвращаем сколько символов мы записали в буфер
        return bufferPos;
    }

    /**
     * Заменяет в буфере текущее слово на знак вопроса, если оно число
     * @param buffer буфер с записанным запросом
     * @param beginTokenBufferPos индекс начала слова в буфере
     * @param bufferPos текущий конец буфера
     * @return новый конец буфера
     */
    private static int changeIfNumber(char[] buffer, int beginTokenBufferPos, int bufferPos) {
        // Число либо начинается на цифру, либо на минус, а следующий символ за ним - цифра,
        // при этом слово из одного минуса состоять не может, так как тогда это не слово, а разделитель.
        boolean isNumber = (Character.isDigit(buffer[beginTokenBufferPos]) ||
                (buffer[beginTokenBufferPos] == '-' && Character.isDigit(buffer[beginTokenBufferPos + 1])));

        if (isNumber) {
            // Если слово - число, заменяем его на знак вопроса
            buffer[beginTokenBufferPos++] = '?';
            bufferPos = beginTokenBufferPos;
        }
        return bufferPos;
    }

    /**
     * Добавляет строку в буфер
     * @param buffer буфер, куда нужно добавить строку
     * @param str строка для добавления
     * @param bufferPos текущая позиция в буфере
     * @return новый конец буфера
     */
    private static int appendString(char[] buffer, String str, int bufferPos) {
        for (int i = 0; i < str.length(); i++) {
            buffer[bufferPos++] = str.charAt(i);
        }
        return bufferPos;
    }

    /**
     * Определяет стадию обработки текущего значимого символа запроса
     * @param c текущий символ
     * @param nextChar следующий символ
     * @param tokenStageWasNotInWord текущая стадия и предыдущая непробельная стадия обработки символов запроса
     *                               не являются стадиями обработки слова
     * @return Стадия обработки текущего значимого символа запроса
     */
    private static TokenStateEnum getTokenStateByChar(
            char c, char nextChar, boolean tokenStageWasNotInWord) {
        if (Character.isWhitespace(c)) {
            // это пробельный символ, все просто
            return TokenStateEnum.IN_WHITESPACES;
        }

        if (c == '(' || c == ')') {
            // это открывающая или закрывающая скобка, тоже думать нечего
            return TokenStateEnum.IN_BRACES;
        }

        if (isNotPotentialDelimiter(c)) {
            // Это не символ разделителя, значит остается только слово
            return TokenStateEnum.IN_WORD;
        }

        if (tokenStageWasNotInWord && c == '-' && Character.isDigit(nextChar)) {
            // Если в предыдущей непробельной стадии, или в текущей стадии обрабатывали не слово,
            // а скобки или разделители, и вдруг встретили минус, за которым идет число, то этот минус относится к числу
            return TokenStateEnum.IN_WORD;
        }
        // Теперь точно стадия обработки разделителей
        return TokenStateEnum.IN_DELIMITERS;
    }

    /**
     * Проверяет, не является ли текущий символ - потенциальным символом разделителем из набора "<,>=-+*%/!^~|"
     * @param c текущий символ
     * @return true, если текущий символ c - точно не разделитель, false - если текущй символ - возможно разделитель
     */
    public static boolean isNotPotentialDelimiter(char c) {
        // Проверяем бит разделителя в заранее подготовленных битовых масках
        if (c < 64) {
            return (delimiters1 & (1L << c)) == 0;
        }
        return (delimiters2 & (1L << (c - 64))) == 0;
    }

    /**
     * Стадия обработки запроса
     */
    private enum ParseStateEnum {
        /**
         * Мы внутри кавычек
         */
        IN_QUOTA,

        /**
         * Мы внутри блока in (...)
         */
        IN_IN,

        /**
         * Мы внутри блока комментария &#47;&#42; &#42;&#47;
         */
        IN_MULTILINE_COMMENT,

        /**
         * Мы внутри блока комментария, начинающегося с --;
         */
        IN_SINGLE_LINE_COMMENT,

        /**
         * Мы обрабатываем значимые символы запроса
         */
        IN_TOKEN
    }

    /**
     * Стадия обработки значимых символов запроса
     */
    private enum TokenStateEnum {
        /**
         * Мы обрабатываем слово
         */
        IN_WORD,

        /**
         * Мы обрабатываем последовательность символов-разделителей "<,>=-+*%/!^~|"
         */
        IN_DELIMITERS,

        /**
         * Мы обрабатываем последовательность непробельных символов
         */
        IN_WHITESPACES,

        /**
         * Мы обрабатываем последовательность открывающих и закрывающих круглых скобок
         */
        IN_BRACES
    }

}
