package ru.yandex.direct.staff.client;

import java.time.LocalDate;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.google.common.net.HttpHeaders;
import one.util.streamex.StreamEx;

import ru.yandex.direct.asynchttp.ParallelFetcherFactory;
import ru.yandex.direct.asynchttp.Result;
import ru.yandex.direct.http.smart.core.Call;
import ru.yandex.direct.http.smart.core.Smart;
import ru.yandex.direct.staff.client.model.GapType;
import ru.yandex.direct.staff.client.model.StaffConfiguration;
import ru.yandex.direct.staff.client.model.json.Gap;
import ru.yandex.direct.staff.client.model.json.GapsSearchResponse;
import ru.yandex.direct.staff.client.model.json.PersonInfo;
import ru.yandex.direct.staff.client.model.json.StaffRawInfo;
import ru.yandex.direct.tvm.TvmIntegration;
import ru.yandex.direct.tvm.TvmService;

import static ru.yandex.direct.http.smart.error.ErrorUtils.checkResultForErrors;
import static ru.yandex.direct.staff.client.model.json.StaffRawInfo.DEFAULT_FIELDS;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;

/**
 * Клиент к публичному api staff https://wiki.yandex-team.ru/staff/apiv3/
 */
public class StaffClient {
    public static final int MAX_LIMIT = 1000;
    public static final String DATE_FORMAT = "yyyy-MM-dd'T'HH:mm:ss";

    private final StaffV3Api staffV3Api;
    private final GapsApi gapsApi;

    public StaffClient(StaffConfiguration staffConfiguration,
                       ParallelFetcherFactory parallelFetcherFactory, TvmIntegration tvmIntegration, boolean isProd) {
        Smart.Builder builder = Smart.builder()
                .withParallelFetcherFactory(parallelFetcherFactory)
                .withProfileName("staff_v3_api")
                .withBaseUrl(staffConfiguration.getStaffApiUrl())
                .addHeaderConfigurator(headers -> headers.add("Content-type", "application/json"));
        if (staffConfiguration.getOauthToken() != null) {
            builder.addHeaderConfigurator(headers -> headers.add("Authorization",
                    "OAuth " + staffConfiguration.getOauthToken()));
        } else {
            builder.useTvm(tvmIntegration,
                    isProd ? TvmService.STAFF_PROD
                            : TvmService.STAFF_TEST);
        }

        staffV3Api = builder.build().create(StaffV3Api.class);

        gapsApi = Smart.builder()
                .withParallelFetcherFactory(parallelFetcherFactory)
                .withProfileName("gaps_api")
                .withBaseUrl(staffConfiguration.getGapsApiUrl())
                .addHeaderConfigurator(headers -> headers.add(HttpHeaders.CONTENT_TYPE, "application/json"))
                .addHeaderConfigurator(headers -> headers.add(HttpHeaders.AUTHORIZATION,
                        "OAuth " + staffConfiguration.getOauthToken()))
                .build().create(GapsApi.class);
    }

    public Map<String, PersonInfo> getStaffUserInfos(Collection<String> logins) {
        String comaSeparatedLogins = String.join(",", logins);
        List<PersonInfo> personInfos = getPersonsByLogins(comaSeparatedLogins, DEFAULT_FIELDS);
        return listToMap(personInfos, PersonInfo::getLogin);
    }

    public Set<String> getExistingUsers(Collection<String> logins) {
        String comaSeparatedLogins = String.join(",", logins);
        List<PersonInfo> personInfos = getPersonsByLogins(comaSeparatedLogins, "login");
        return listToSet(personInfos, PersonInfo::getLogin);
    }

    private List<PersonInfo> getPersonsByLogins(String comaSeparatedLogins, String fields) {
        Call<StaffRawInfo> call = staffV3Api.getPersonsByLogins(comaSeparatedLogins, fields);
        Result<StaffRawInfo> result = call.execute();
        checkResultForErrors(result, StaffApiException::new);
        return result.getSuccess().getPersonInfo();
    }

    public List<PersonInfo> getStaffUsers(String query) {
        return getStaffUsers(query, DEFAULT_FIELDS);
    }

    public List<PersonInfo> getStaffUsers(String query, String fields) {
        Call<StaffRawInfo> call = staffV3Api.getPersonsByQuery(query, fields, false, MAX_LIMIT);
        Result<StaffRawInfo> result = call.execute();
        checkResultForErrors(result, StaffApiException::new);
        return result.getSuccess().getPersonInfo();
    }

    public Map<String, List<Gap>> findGaps(LocalDate dateFrom, LocalDate dateTo, Set<String> logins) {
        Result<GapsSearchResponse> result = gapsApi.getPersonsByQuery(dateFrom, dateTo, logins,
                Set.of(GapType.ILLNESS, GapType.VACATION, GapType.PAID_DAY_OFF), MAX_LIMIT).execute();
        checkResultForErrors(result, StaffApiException::new);
        return StreamEx.of(result.getSuccess().getGaps())
                .groupingBy(Gap::getPersonLogin);
    }
}
