package ru.yandex.direct.telegram.client

import org.asynchttpclient.AsyncHttpClient
import ru.yandex.direct.asynchttp.FetcherSettings
import ru.yandex.direct.asynchttp.ParallelFetcherFactory
import ru.yandex.direct.http.smart.converter.RequestConverterFactory
import ru.yandex.direct.http.smart.converter.ResponseConverterFactory
import ru.yandex.direct.http.smart.core.Smart
import ru.yandex.direct.liveresource.LiveResource
import ru.yandex.direct.telegram.client.api.ChatAction
import ru.yandex.direct.telegram.client.api.GetUpdatesRequest
import ru.yandex.direct.telegram.client.api.Markup
import ru.yandex.direct.telegram.client.api.ParseMode
import ru.yandex.direct.telegram.client.api.SendChatActionRequest
import ru.yandex.direct.telegram.client.api.SendMessageRequest
import ru.yandex.direct.telegram.client.api.TelegramApi
import ru.yandex.direct.telegram.client.api.executeChecked

class TelegramClient(
    asyncHttpClient: AsyncHttpClient,
    telegramBotToken: LiveResource,
) {
    companion object {
        private const val BASE_URL = "https://api.telegram.org"
    }

    private val telegramApi: TelegramApi = Smart.builder()
        .withRequestConverterFactory(
            RequestConverterFactory(
                mapOf(TelegramRequestConverter::class.java to TelegramRequestConverter<Any>()),
            )
        )
        .withResponseConverterFactory(
            ResponseConverterFactory.builder()
                .addConverters(TelegramResponseConverter())
                .build()
        )
        .withParallelFetcherFactory(ParallelFetcherFactory(asyncHttpClient, FetcherSettings()))
        .withProfileName("telegram_client")
        .withBaseUrl("$BASE_URL/bot${telegramBotToken.content}")
        .build()
        .create(TelegramApi::class.java)

    val api: TelegramApi = telegramApi

    fun getMe() = telegramApi.getMe().executeChecked()

    fun sendMessage(
        chatId: Long,
        text: String,
        parseMode: ParseMode? = null,
        disableWebPagePreview: Boolean? = null,
        disableNotification: Boolean? = null,
        replyToMessageId: Int? = null,
        replyMarkup: Markup? = null,
    ) = telegramApi.sendMessage(
        SendMessageRequest(
            chatId = chatId,
            text = text,
            parseMode = parseMode,
            disableNotification = disableNotification,
            disableWebPagePreview = disableWebPagePreview,
            replyToMessageId = replyToMessageId,
            replyMarkup = replyMarkup,
        )
    ).executeChecked()

    fun sendChatAction(chatId: Long, action: ChatAction) =
        telegramApi.sendChatAction(
            SendChatActionRequest(
                chatId = chatId,
                action = action,
            )
        ).executeChecked()

    fun getUpdates(
        offset: Int? = null,
        limit: Int? = null,
        timeout: Int? = null,
        allowedUpdates: List<String>? = null,
    ) = telegramApi.getUpdates(
        GetUpdatesRequest(
            offset = offset,
            limit = limit,
            timeout = timeout,
            allowedUpdates = allowedUpdates,
        )
    ).executeChecked()
}
