package ru.yandex.direct.telephony.client;

import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import io.grpc.Metadata;

import ru.yandex.direct.telephony.client.model.TelephonyPhoneRequest;
import ru.yandex.direct.tvm.TvmIntegration;
import ru.yandex.telephony.backend.lib.proto.telephony_platform.EntityIDs;
import ru.yandex.telephony.backend.lib.proto.telephony_platform.ServiceNumberKind;
import ru.yandex.telephony.backend.lib.proto.telephony_platform.ServiceNumberRouting;
import ru.yandex.telephony.backend.lib.proto.telephony_platform.ServiceNumberState;
import ru.yandex.telephony.backend.lib.proto.telephony_platform.ServiceNumbersRequest;
import ru.yandex.telephony.backend.lib.proto.telephony_platform.UpdateServiceNumberData;
import ru.yandex.telephony.backend.lib.proto.telephony_platform.UpdateServiceNumberRequest;
import ru.yandex.telephony.backend.lib.proto.telephony_platform.UpdateServiceNumberWithVersionRequest;

import static org.apache.commons.lang3.StringUtils.isEmpty;

public class ProtobufMapper {

    public static final String CLIENT_ID_META_KEY = "clientID";
    public static final String COUNTER_ID_META_KEY = "counterID";
    public static final String ORG_ID_META_KEY = "orgID";
    public static final String CALL_SOURCE = "callSource";

    static final Metadata.Key<String> REQUEST_ID_KEY =
            Metadata.Key.of("X-Request-Id", Metadata.ASCII_STRING_MARSHALLER);
    static final Metadata.Key<String> YA_SERVICE_TICKET_KEY =
            Metadata.Key.of(TvmIntegration.SERVICE_TICKET_HEADER, Metadata.ASCII_STRING_MARSHALLER);
    static final Metadata.Key<String> TELEPHONY_TICKET_KEY =
            Metadata.Key.of("X-Telephony-Ticket-Id", Metadata.ASCII_STRING_MARSHALLER);

    static Metadata createMeta(String telephonyTicket, String tvmTicket) {
        Metadata headers = new Metadata();
        headers.put(TELEPHONY_TICKET_KEY, telephonyTicket);
        headers.put(REQUEST_ID_KEY, UUID.randomUUID().toString());
        headers.put(YA_SERVICE_TICKET_KEY, tvmTicket);
        return headers;
    }

    /**
     * Запрос для получения свободных номеров. Можно запросить несколько номеров, указав {@code setCount(n)}
     */
    static ServiceNumbersRequest createServiceNumbersRequest(@Nullable String regionCode) {
        ServiceNumberKind numberKind = isEmpty(regionCode) ? ServiceNumberKind.DEF : ServiceNumberKind.ABC;
        ServiceNumbersRequest.ServiceNumbersFilter.Builder builder =
                ServiceNumbersRequest.ServiceNumbersFilter.newBuilder()
                        .setState(ServiceNumberState.ACTIVE)
                        .setKind(numberKind)
                        .addAllAbsentMetaKeys(List.of(CLIENT_ID_META_KEY))
                        .setCount(1);
        if (regionCode != null) {
            builder.setRegionCode(regionCode);
        }
        ServiceNumbersRequest.ServiceNumbersFilter filter = builder.build();
        return ServiceNumbersRequest.newBuilder()
                .setFilter(filter)
                .build();
    }

    static ServiceNumbersRequest createServiceNumbersByIdRequest(Collection<String> telephonyServiceIds) {
        EntityIDs numberIDs = EntityIDs.newBuilder()
                .addAllIds(telephonyServiceIds)
                .build();
        return ServiceNumbersRequest.newBuilder()
                .setServiceNumberIDs(numberIDs)
                .build();
    }

    /**
     * Запрос для получения привязанных номеров клиента
     */
    static ServiceNumbersRequest createServiceNumbersRequest(long clientId) {
        Map<String, String> meta = Map.of(CLIENT_ID_META_KEY, String.valueOf(clientId));
        ServiceNumbersRequest.ServiceNumbersFilter filter =
                ServiceNumbersRequest.ServiceNumbersFilter.newBuilder()
                        .putAllMeta(meta)
                        .build();
        return ServiceNumbersRequest.newBuilder()
                .setFilter(filter)
                .build();
    }

    /**
     * Запрос для получения пачки номеров директа без каких либо фильтраций
     */
    static ServiceNumbersRequest createServiceNumbersBatchRequest(int offset, int count) {
        ServiceNumbersRequest.ServiceNumbersFilter filter =
                ServiceNumbersRequest.ServiceNumbersFilter.newBuilder()
                        .setOffset(offset)
                        .setCount(count)
                        .build();

        return ServiceNumbersRequest.newBuilder()
                .setFilter(filter)
                .build();
    }

    /**
     * Запрос для привязки номера телефонии к клиенту.
     * В {@code TelephonyPhoneRequest#telephonyServiceId} указывается id,
     * полученный при запросе {@link TelephonyGrpcApi#getServiceNumber}
     */
    static UpdateServiceNumberRequest createUpdateServiceNumberRequest(long clientId,
                                                                       TelephonyPhoneRequest telephonyPhone,
                                                                       String playbackId) {
        UpdateServiceNumberData dataRequest = createUpdateData(clientId, telephonyPhone, playbackId);
        return UpdateServiceNumberRequest.newBuilder()
                .setServiceNumberID(telephonyPhone.getTelephonyServiceId())
                .setUpdateData(dataRequest)
                .build();
    }

    /**
     * Запрос для привязки номера телефонии к клиенту с учетом версии.
     * В {@code TelephonyPhoneRequest#telephonyServiceId} указывается id,
     * полученный при запросе {@link TelephonyGrpcApi#getServiceNumber}
     */
    static UpdateServiceNumberWithVersionRequest createUpdateServiceNumberRequest(
            long clientId,
            TelephonyPhoneRequest telephonyPhone,
            String playbackId,
            int version) {
        UpdateServiceNumberData dataRequest = createUpdateData(clientId, telephonyPhone, playbackId);
        return UpdateServiceNumberWithVersionRequest.newBuilder()
                .setServiceNumberID(telephonyPhone.getTelephonyServiceId())
                .setUpdateData(dataRequest)
                .setVersion(version)
                .build();
    }

    private static UpdateServiceNumberData createUpdateData(
            long clientId,
            TelephonyPhoneRequest telephonyPhone,
            String playbackId) {
        ServiceNumberRouting.CallHandler.ExternalCallHandler.Builder externalCallHandlerBuilder =
                ServiceNumberRouting.CallHandler.ExternalCallHandler.newBuilder()
                .setExternalNum(telephonyPhone.getRedirectPhone())
                .setBeforeConnectedPlaybackID(telephonyPhone.getBeforeConnectedPlaybackId() != null
                        ? telephonyPhone.getBeforeConnectedPlaybackId()
                        : playbackId);

        if (telephonyPhone.getBeforeConversationPlaybackID() != null) {
            externalCallHandlerBuilder.setBeforeConversationPlaybackID(
                    telephonyPhone.getBeforeConversationPlaybackID()
            );
        }

        ServiceNumberRouting.CallHandler.ExternalCallHandler externalCallHandler = externalCallHandlerBuilder.build();

        ServiceNumberRouting.CallHandler handler = ServiceNumberRouting.CallHandler.newBuilder()
                .setExternalHandler(externalCallHandler)
                .build();

        ServiceNumberRouting.Builder routingBuilder = ServiceNumberRouting.newBuilder()
                .setHandler(handler);

        if (telephonyPhone.getPromptPlaybackId() != null) {
            routingBuilder.addPromptPlaybackIDs(telephonyPhone.getPromptPlaybackId());
        }

        Map<String, String> meta = composePhoneMeta(clientId, telephonyPhone);

        UpdateServiceNumberData.Builder dataBuilder = UpdateServiceNumberData.newBuilder()
                .setRouting(routingBuilder.build())
                .putAllMeta(meta);

        if (telephonyPhone.getDisableProxyBridgedCallerNum() != null) {
            dataBuilder.setDisableProxyBridgedCallerNum(telephonyPhone.getDisableProxyBridgedCallerNum());
        }

        if (telephonyPhone.getCallRecordingScope() != null) {
            dataBuilder.setCallRecordingScope(telephonyPhone.getCallRecordingScope());
        }

        return dataBuilder.build();
    }

    @Nonnull
    static Map<String, String> composePhoneMeta(long clientId, TelephonyPhoneRequest telephonyPhone) {
        Map<String, String> meta = new HashMap<>();
        meta.put(CLIENT_ID_META_KEY, String.valueOf(clientId));
        meta.put(COUNTER_ID_META_KEY, telephonyPhone.getCounterId().toString());
        meta.put(CALL_SOURCE, telephonyPhone.getCallSource().getName());
        Long permalinkId = telephonyPhone.getPermalinkId();
        if (permalinkId != null && permalinkId != 0) { // у подменников на сайте нет связанной организации
            meta.put(ORG_ID_META_KEY, Long.toString(permalinkId));
        }
        return meta;
    }

    /**
     * Запрос для отвязки телефона от клиента.
     */
    static UpdateServiceNumberRequest createUpdateServiceNumberRequest(
            String serviceNumberId,
            boolean shouldSendToQuarantine
    ) {
        ServiceNumberRouting.CallHandler handler = ServiceNumberRouting.CallHandler.newBuilder()
                .setDisconnectHandler(ServiceNumberRouting.CallHandler.DisconnectCallHandler.newBuilder().build())
                .build();
        ServiceNumberRouting routing = ServiceNumberRouting.newBuilder()
                .setHandler(handler).build();
        UpdateServiceNumberData dataRequest = UpdateServiceNumberData.newBuilder()
                .setRouting(routing)
                .putAllMeta(Map.of())
                .setShouldSendToQuarantine(shouldSendToQuarantine)
                .build();
        return UpdateServiceNumberRequest.newBuilder()
                .setServiceNumberID(serviceNumberId)
                .setUpdateData(dataRequest)
                .build();
    }
}
