package ru.yandex.direct.telephony.client;

import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.function.Supplier;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import io.grpc.ClientInterceptor;
import io.grpc.ManagedChannel;
import io.grpc.ManagedChannelBuilder;
import io.grpc.Metadata;
import io.grpc.StatusRuntimeException;
import org.asynchttpclient.request.body.multipart.ByteArrayPart;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.asynchttp.ParallelFetcherFactory;
import ru.yandex.direct.asynchttp.Result;
import ru.yandex.direct.http.smart.core.Smart;
import ru.yandex.direct.telephony.client.model.GetTicketResponse;
import ru.yandex.direct.telephony.client.model.PlaybackUploadResponse;
import ru.yandex.direct.telephony.client.model.TelephonyPhoneRequest;
import ru.yandex.direct.tvm.TvmIntegration;
import ru.yandex.direct.tvm.TvmService;
import ru.yandex.telephony.backend.lib.proto.telephony_platform.ServiceNumber;
import ru.yandex.telephony.backend.lib.proto.telephony_platform.ServiceNumbersList;
import ru.yandex.telephony.backend.lib.proto.telephony_platform.ServiceNumbersRequest;
import ru.yandex.telephony.backend.lib.proto.telephony_platform.TelephonyPlatformServiceGrpc;
import ru.yandex.telephony.backend.lib.proto.telephony_platform.UpdateServiceNumberRequest;
import ru.yandex.telephony.backend.lib.proto.telephony_platform.UpdateServiceNumberWithVersionRequest;
import ru.yandex.telephony.backend.lib.proto.telephony_platform.UpdateServiceSettingsRequest;

import static io.grpc.stub.MetadataUtils.newAttachHeadersInterceptor;
import static ru.yandex.direct.http.smart.error.ErrorUtils.checkResultForErrors;
import static ru.yandex.direct.telephony.client.ProtobufMapper.REQUEST_ID_KEY;
import static ru.yandex.direct.telephony.client.ProtobufMapper.createMeta;
import static ru.yandex.direct.telephony.client.ProtobufMapper.createServiceNumbersBatchRequest;
import static ru.yandex.direct.telephony.client.ProtobufMapper.createServiceNumbersByIdRequest;
import static ru.yandex.direct.telephony.client.ProtobufMapper.createServiceNumbersRequest;
import static ru.yandex.direct.telephony.client.ProtobufMapper.createUpdateServiceNumberRequest;

/**
 * Клиент к API Телефонии
 */
public class TelephonyClient implements TelephonyGrpcApi {

    private static final Logger logger = LoggerFactory.getLogger(TelephonyClient.class);

    private final SmartApi api;
    private final Supplier<String> tvmTicketSupplier;
    private final Supplier<String> playbackIdSupplier;

    private List<ClientInterceptor> clientInterceptors;

    public TelephonyClient(String url,
                           TvmIntegration tvmIntegration,
                           TvmService tvmService,
                           ParallelFetcherFactory parallelFetcherFactory,
                           Supplier<String> playbackIdSupplier) {
        this.api = createApi(url, tvmIntegration, tvmService, parallelFetcherFactory);
        this.tvmTicketSupplier = () -> tvmIntegration.getTicket(tvmService);
        this.playbackIdSupplier = playbackIdSupplier;
    }

    private SmartApi createApi(String url, TvmIntegration tvmIntegration, TvmService tvmService,
                               ParallelFetcherFactory parallelFetcherFactory) {
        return Smart.builder()
                .withParallelFetcherFactory(parallelFetcherFactory)
                .withProfileName("telephony")
                .useTvm(tvmIntegration, tvmService)
                .withBaseUrl(url)
                .build()
                .create(SmartApi.class);
    }

    @Override
    public ServiceNumber getServiceNumber(@Nullable String regionCode) {
        ServiceNumbersRequest request = createServiceNumbersRequest(regionCode);
        GetTicketResponse ticket = getTicket();
        ManagedChannel channel = getManagedChannel(ticket);
        var blockingStub = getBlockingStub(channel, ticket.getTicketId());
        try {
            logger.info("Started getServiceNumber request: {}", request);
            ServiceNumbersList serviceNumbers = blockingStub.getServiceNumbers(request);
            logger.info("Telephony getServiceNumber response: [{}]", serviceNumbers);
            List<ServiceNumber> serviceNumberList = serviceNumbers.getNumbersList();

            if (serviceNumberList.isEmpty()) {
                logger.error("Got empty phone numbers list");
                return null;
            } else {
                return serviceNumberList.get(0);
            }

        } catch (StatusRuntimeException ex) {
            logger.error("Request were interrupted: {} ", ex.getStatus());
            throw new TelephonyClientException(ex);
        } finally {
            channel.shutdown();
        }
    }

    /**
     * Получить один номер Телефонии с заданным кодом города.
     * Если свободных номеров с кодом {@code regionCode} нет, то делается запрос за DEF-номером.
     */
    @Nullable
    public ServiceNumber tryToGetServiceNumber(@Nonnull String regionCode) {
        logger.info("Try to request phone number with region code {}", regionCode);
        var serviceNumber = getServiceNumber(regionCode);
        if (serviceNumber != null) {
            logger.info("Successfully requested phone number with region code {}", regionCode);
            return serviceNumber;
        }
        logger.info("Failed to request phone number with region code {}. DEF phone number will be used", regionCode);
        return getServiceNumber();
    }

    @Override
    public ServiceNumber getServiceNumber() {
        return getServiceNumber(null);
    }

    @Override
    public List<ServiceNumber> getClientServiceNumbers(long clientId) {
        ServiceNumbersRequest request = createServiceNumbersRequest(clientId);
        GetTicketResponse ticket = getTicket();
        ManagedChannel channel = getManagedChannel(ticket);
        var blockingStub = getBlockingStub(channel, ticket.getTicketId());
        try {
            logger.info("Started getClientServiceNumbers request: {}", request);
            ServiceNumbersList clientNumbers = blockingStub.getServiceNumbers(request);
            logger.info("Telephony getClientServiceNumbers response: [{}]", clientNumbers);
            return clientNumbers.getNumbersList();
        } catch (StatusRuntimeException ex) {
            logger.error("Request were interrupted: {}", ex.getStatus());
            throw new TelephonyClientException(ex);
        } finally {
            channel.shutdown();
        }
    }

    @Override
    public List<ServiceNumber> getServiceNumbersBatch(int offset, int count) {
        ServiceNumbersRequest request = createServiceNumbersBatchRequest(offset, count);
        GetTicketResponse ticket = getTicket();
        ManagedChannel channel = getManagedChannel(ticket);
        var blockingStub = getBlockingStub(channel, ticket.getTicketId());
        try {
            logger.info("Started getServiceNumbersBatch request: {}", request);
            ServiceNumbersList clientNumbers = blockingStub.getServiceNumbers(request);
            List<ServiceNumber> result = clientNumbers.getNumbersList();
            logger.info("Telephony getServiceNumbersBatch numbers count: [{}]", result.size());
            return result;
        } catch (StatusRuntimeException ex) {
            logger.error("Request were interrupted: {}", ex.getStatus());
            throw new TelephonyClientException(ex);
        } finally {
            channel.shutdown();
        }
    }

    @Override
    public List<ServiceNumber> getServiceNumberByIds(Collection<String> telephonyServiceIds) {
        ServiceNumbersRequest request = createServiceNumbersByIdRequest(telephonyServiceIds);
        GetTicketResponse ticket = getTicket();
        ManagedChannel channel = getManagedChannel(ticket);
        var blockingStub = getBlockingStub(channel, ticket.getTicketId());
        try {
            logger.info("Started getClientServiceNumbers request: {}", request);
            ServiceNumbersList serviceNumbers = blockingStub.getServiceNumbers(request);
            logger.info("Telephony getClientServiceNumbers response: [{}]", serviceNumbers);
            return serviceNumbers.getNumbersList();
        } catch (StatusRuntimeException ex) {
            logger.error("Request were interrupted: {}", ex.getStatus());
            throw new TelephonyClientException(ex);
        } finally {
            channel.shutdown();
        }
    }

    @Override
    public void linkServiceNumber(long clientId, TelephonyPhoneRequest telephonyPhone) {
        String playbackId = playbackIdSupplier.get();
        if (playbackId.isEmpty()) {
            throw new TelephonyClientException("Playback is empty");
        }
        GetTicketResponse ticket = getTicket();
        ManagedChannel channel = getManagedChannel(ticket);
        var blockingStub = getBlockingStub(channel, ticket.getTicketId());
        try {
            UpdateServiceNumberRequest request = createUpdateServiceNumberRequest(clientId, telephonyPhone, playbackId);
            logger.info("Started linkServiceNumber request: {}", request);
            //noinspection ResultOfMethodCallIgnored
            blockingStub.updateServiceNumber(request);
            logger.info("Successfully link service number");
        } catch (StatusRuntimeException ex) {
            logger.error("Request were interrupted: {} ", ex.getStatus());
            throw new TelephonyClientException(ex);
        } finally {
            channel.shutdown();
        }
    }

    @Override
    public void linkServiceNumber(long clientId, TelephonyPhoneRequest telephonyPhone, int version) {
        String playbackId = playbackIdSupplier.get();
        if (playbackId.isEmpty()) {
            throw new TelephonyClientException("Playback is empty");
        }
        GetTicketResponse ticket = getTicket();
        ManagedChannel channel = getManagedChannel(ticket);
        var blockingStub = getBlockingStub(channel, ticket.getTicketId());
        try {
            UpdateServiceNumberWithVersionRequest request =
                    createUpdateServiceNumberRequest(clientId, telephonyPhone, playbackId, version);
            logger.info("Started linkServiceNumber request: {}", request);
            //noinspection ResultOfMethodCallIgnored
            blockingStub.updateServiceNumberWithVersion(request);
            logger.info("Successfully link service number");
        } catch (StatusRuntimeException ex) {
            logger.error("Request were interrupted: {} ", ex.getStatus());
            throw new TelephonyClientException(ex);
        } finally {
            channel.shutdown();
        }
    }

    @Override
    public void unlinkServiceNumber(String serviceNumberId, boolean shouldSendToQuarantine) {
        UpdateServiceNumberRequest request = createUpdateServiceNumberRequest(serviceNumberId, shouldSendToQuarantine);
        GetTicketResponse ticket = getTicket();
        ManagedChannel channel = getManagedChannel(ticket);
        var blockingStub = getBlockingStub(channel, ticket.getTicketId());
        try {
            logger.info("Started unlinkServiceNumber request: {}", request);
            //noinspection ResultOfMethodCallIgnored
            blockingStub.updateServiceNumber(request);
            logger.info("Successfully unlink service number");
        } catch (StatusRuntimeException ex) {
            logger.error("Request were interrupted: {} ", ex.getStatus());
            throw new TelephonyClientException(ex);
        } finally {
            channel.shutdown();
        }
    }

    @Override
    public void updateServiceSettings(UpdateServiceSettingsRequest updateSettings) {
        GetTicketResponse ticket = getTicket();
        ManagedChannel channel = getManagedChannel(ticket);
        var blockingStub = getBlockingStub(channel, ticket.getTicketId());
        try {
            logger.info("Started updateServiceSettings request: {}", updateSettings);
            //noinspection ResultOfMethodCallIgnored
            blockingStub.updateServiceSettings(updateSettings);
            logger.info("updateServiceSettings successfully called");
        } catch (StatusRuntimeException ex) {
            logger.error("Request were interrupted: {} ", ex.getStatus());
            throw new TelephonyClientException(ex);
        } finally {
            channel.shutdown();
        }
    }

    GetTicketResponse getTicket() {
        Result<GetTicketResponse> execute = api.tickets().execute();
        checkResultForErrors(execute, TelephonyClientException::new);
        return execute.getSuccess();
    }

    private ManagedChannel getManagedChannel(GetTicketResponse ticket) {
        return ManagedChannelBuilder
                .forAddress(ticket.getApiHost(), ticket.getApiPort())
                .usePlaintext()
                .build();
    }

    private TelephonyPlatformServiceGrpc.TelephonyPlatformServiceBlockingStub getBlockingStub(ManagedChannel channel,
                                                                                              String ticketId) {
        var blockingStub = TelephonyPlatformServiceGrpc.newBlockingStub(channel);
        Metadata metadata = createMeta(ticketId, tvmTicketSupplier.get());
        logger.info("Starting request with id: {}", metadata.get(REQUEST_ID_KEY));
        return blockingStub.withInterceptors(getClientInterceptors(metadata));
    }

    @Nonnull
    private ClientInterceptor[] getClientInterceptors(Metadata metadata) {
        var list = new ArrayList<ClientInterceptor>();
        list.add(newAttachHeadersInterceptor(metadata));
        if (clientInterceptors != null) {
            list.addAll(clientInterceptors);
        }
        return list.toArray(ClientInterceptor[]::new);
    }

    /**
     * <a href="https://wiki.yandex-team.ru/telephony/telephony-platform-calltracking/#post/playbacks">
     * Загрузить плейбек (аудиофайл) в Телефонию</a>
     */
    public PlaybackUploadResponse uploadPlayback(byte[] data) {
        logger.info("Uploading playback...");
        var playbackData =
                new ByteArrayPart("content", data, "application/octet-stream", StandardCharsets.UTF_8, "playback.wav");
        Result<PlaybackUploadResponse> result = api.playbacks(playbackData).execute();
        checkResultForErrors(result, TelephonyClientException::new);
        PlaybackUploadResponse response = result.getSuccess();
        logger.info("Uploaded playback: playbackId = {}", response.getPlaybackId());
        return response;
    }

    public void setClientInterceptors(List<ClientInterceptor> clientInterceptors) {
        this.clientInterceptors = clientInterceptors;
    }
}
