package ru.yandex.direct.libs.timetarget;

import java.util.Arrays;
import java.util.Map;

import static com.google.common.base.Preconditions.checkArgument;

/**
 * Определяет коэффициенты временного таргетинга для одного дня.
 * <p>
 * Эквивалентно {@link Map}'е {@code int->int} с {@code 0} вместо {@code null}.
 */
@SuppressWarnings("WeakerAccess")
public class HoursCoef {
    private final short[] hourToPercent;

    public HoursCoef() {
        this(new short[24]);
    }

    private HoursCoef(short[] hourToPercent) {
        this.hourToPercent = hourToPercent;
    }

    /**
     * Задаёт новое значение коэффициента для указанного часа из отрезка {@code [0-23]}.
     */
    public void setCoef(int hour, int coef) {
        checkArgument(0 <= hour && hour <= 23, "hour must be in range [0; 23]");
        checkArgument(0 <= coef && coef <= 200, "hour timetarget coef must be in range [0; 200]");
        hourToPercent[hour] = (short) coef;
    }

    public HoursCoef withHourCoef(int hour, int coef) {
        setCoef(hour, coef);
        return this;
    }

    /**
     * @param hour час (0-23)
     * @return коэффициент в процентах в диапазоне 0-200
     */
    public int getCoefForHour(int hour) {
        return hourToPercent[hour];
    }

    /**
     * @return глубокая копия объекта
     */
    public HoursCoef copy() {
        return new HoursCoef(Arrays.copyOf(hourToPercent, hourToPercent.length));
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        HoursCoef hoursCoef = (HoursCoef) o;
        return Arrays.equals(hourToPercent, hoursCoef.hourToPercent);
    }

    @Override
    public int hashCode() {
        return Arrays.hashCode(hourToPercent);
    }

    @Override
    public String toString() {
        return "HoursCoef{" +
                "hourToPercent=" + Arrays.toString(hourToPercent) +
                '}';
    }
}
