package ru.yandex.direct.libs.timetarget;

import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.regex.Pattern;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkState;
import static ru.yandex.direct.libs.timetarget.TimeTargetUtils.defaultTargetingHoursCoefs;
import static ru.yandex.direct.libs.timetarget.TimeTargetUtils.isZeroCoefs;

/**
 * Представление настроек временного таргетинга в Директе.
 * <p>
 * Может быть сконструирован из строкового представления с помощью {@link #parseRawString(String)}.
 * Для обратного преобразования предназначен метод {@link #toRawFormat()}
 */
@SuppressWarnings("WeakerAccess")
public class TimeTarget {

    private static final Pattern TIME_TARGET_FORMAT = Pattern.compile("^([1-9]([A-X][a-u]?){0,24}){0,9}(;.*+)?");
    public static final char PROP_DELIMITER = ';';

    private Map<WeekdayType, HoursCoef> weekdayCoefs;
    private Preset preset;
    // Исходная строка, считанная из базы, если TimeTarget был получен методом parseRawString
    private String originalTimeTarget;

    public TimeTarget() {
        this.weekdayCoefs = new HashMap<>();
        this.preset = Preset.OTHER;
    }

    /**
     * @param raw Строка формата {@code ([1-9]([A-X][b-jl-u]?){1,24}){1,9}}. Описание в README.md
     * @return {@link TimeTarget}
     */
    public static TimeTarget parseRawString(String raw) {
        checkArgument(TIME_TARGET_FORMAT.matcher(raw).matches(),
                "Wrong format of timeTarget. Expected: ([1-9]([A-X][b-jl-u]?){1,24}){0,9}}(;.*+)?, actual: \"%s\"",
                raw);
        TimeTarget timeTarget = new TimeTarget();
        timeTarget.originalTimeTarget = raw;
        int propDelimPosition = raw.indexOf(PROP_DELIMITER);
        if (propDelimPosition == -1) {
            propDelimPosition = raw.length();
        }

        // разбираем часть до свойств
        int pos = 0;
        WeekdayType wd = null;
        HoursCoef hc = null;
        Integer curHour = null;
        Integer coef = null;
        while (pos < propDelimPosition) {
            char c = raw.charAt(pos);
            if ('0' <= c && c <= '9') {
                // встретили число -- новая группа описания дня
                if (wd != null) {
                    if (curHour != null) {
                        hc.setCoef(curHour, coef);
                    }
                    timeTarget.setWeekdayCoef(wd, hc);
                }
                wd = WeekdayType.getById(c - '0');
                hc = new HoursCoef();
                curHour = null;
                coef = null;
            } else if ('A' <= c && c <= 'X') {
                checkState(wd != null,
                        "Wrong format of timeTarget. Expected: ([1-9]([A-X][b-jl-u]?){1,24}){1,9}}, actual: \"%s\"",
                        raw);
                if (curHour != null) {
                    hc.setCoef(curHour, coef);
                }
                curHour = c - 'A';
                coef = PredefinedCoefs.USUAL.getValue();
            } else if ('a' <= c && c <= 'u') {
                coef = (c - 'a') * 10;
            }
            pos++;
        }

        if (wd != null) {
            if (curHour != null) {
                hc.setCoef(curHour, coef);
            }
            timeTarget.setWeekdayCoef(wd, hc);
        }


        Map<String, String> properties = Collections.emptyMap();
        if (propDelimPosition < raw.length()) {
            // парсим набор свойств вида "key:value", разделённых ';'
            properties = StreamEx.split(raw, PROP_DELIMITER)
                    .skip(1)
                    .nonNull()
                    .remove(String::isEmpty)
                    .map(item -> item.split(":", 2))
                    .mapToEntry(pair -> pair.length > 1 ? pair[1] : "")
                    .mapKeys(pair -> pair[0])
                    .toMap();
        }
        /*
        p - выбранный пользователем preset:
            a - all
            w - worktime
            o - other
        */
        String preset = properties.get("p");
        if (preset != null && !preset.isEmpty()) {
            switch (preset) {
                case "o":
                    timeTarget.setPreset(Preset.OTHER);
                    break;
                case "a":
                    timeTarget.setPreset(Preset.ALL);
                    break;
                case "w":
                    timeTarget.setPreset(Preset.WORKTIME);
                    break;
                default:
                    throw new IllegalArgumentException(
                            String.format("Unexpected value for property 'p': '%s' in timeTarget string '%s'",
                                    preset, raw));
            }
            if (timeTarget.weekdayCoefs.isEmpty()) {
                // заполняем коэффициенты временного таргетинга для случаев,
                // когда в БД указанны одни пресеты (DIRECT-84589)
                timeTarget.setDefaultTimeTargetingCoefs();
            }
        }

        return timeTarget;
    }

    /**
     * Получить глубокую копию объекта
     */
    public TimeTarget copy() {
        return new TimeTarget()
                .withOriginalTimeTarget(originalTimeTarget)
                .withPreset(preset)
                .withWeekdayCoefMap(
                        EntryStream.of(weekdayCoefs).mapValues(HoursCoef::copy).toCustomMap(HashMap::new)
                );
    }

    @Nonnull
    public static String stripProps(@Nonnull String raw) {
        int pos = raw.indexOf(PROP_DELIMITER);
        if (pos == -1) {
            return raw;
        } else {
            return raw.substring(0, pos);
        }
    }

    /**
     * Устанавливает коэффициенты временного таргетинга по-умолчанию: 24 часа, 7 дней в неделю
     */
    private void setDefaultTimeTargetingCoefs() {
        for (int dayOfWeek = 1; dayOfWeek <= 7; dayOfWeek++) {
            weekdayCoefs.put(WeekdayType.getById(dayOfWeek), defaultTargetingHoursCoefs());
        }
    }

    /**
     * @return предыдущее значение настроек для дня, если были. Иначе {@code null}
     */
    public HoursCoef setWeekdayCoef(WeekdayType weekdayType, HoursCoef hoursCoef) {
        return weekdayCoefs.put(weekdayType, hoursCoef);
    }

    public TimeTarget withHourCoef(WeekdayType weekdayType, int hour, int coef) {
        weekdayCoefs.computeIfAbsent(weekdayType, x -> new HoursCoef())
                .setCoef(hour, coef);
        return this;
    }

    public void setWeekdayCoefMap(Map<WeekdayType, HoursCoef> weekdayHoursCoefMap) {
        this.weekdayCoefs = weekdayHoursCoefMap;
    }

    public TimeTarget withWeekdayCoefMap(Map<WeekdayType, HoursCoef> weekdayHoursCoefMap) {
        setWeekdayCoefMap(weekdayHoursCoefMap);
        return this;
    }

    public Map<WeekdayType, HoursCoef> getWeekdayCoefs() {
        return weekdayCoefs;
    }

    @Deprecated  // Ядро игнорирует preset при записи в базу, см. DIRECT-149654
    public TimeTarget withPreset(Preset preset) {
        setPreset(preset);
        return this;
    }

    public Preset getPreset() {
        return preset;
    }

    @Deprecated  // Ядро игнорирует preset при записи в базу, см. DIRECT-149654
    public void setPreset(Preset preset) {
        this.preset = preset;
    }


    public TimeTarget withOriginalTimeTarget(String originalTimeTarget) {
        this.originalTimeTarget = originalTimeTarget;
        return this;
    }

    public String getOriginalTimeTarget() {
        return originalTimeTarget;
    }

    /**
     * Возвращает коэффициенты временного таргетинга для переданного дня
     *
     * @param date               дата, для которого необходимо получить коэффициенты
     * @param productionCalendar реализация производственного календаря {@link ProductionCalendar}, который знает,
     *                           какие будние дни праздничные, а какие выходные рабочие
     */
    @Nullable
    public HoursCoef getHoursCoef(LocalDate date, ProductionCalendar productionCalendar) {
        WeekdayType weekdayType = productionCalendar.getWeekdayType(date);
        HoursCoef hoursCoef = getWeekdayCoefs().get(weekdayType);

        if (weekdayType == WeekdayType.HOLIDAY && hoursCoef == null) {
            //Если для праздника нет коэффициентов, то возвращаем коэффициенты как для обычного дня: DIRECT-87760
            hoursCoef = getWeekdayCoefs().get(getRealWeekdayType(date));
        } else if (weekdayType == WeekdayType.WORKING_WEEKEND) {
            //если это рабочий выходной, получаем коэффициенты для настоящего дня недели.
            hoursCoef = getWeekdayCoefs().get(getRealWeekdayType(date));
            if (weekdayCoefs.containsKey(WeekdayType.WORKING_WEEKEND) && isZeroCoefs(hoursCoef)) {
                //если и у пользователя нет расписания на этот день, но стоит чекбокс "учитывать рабочие выходные"
                //то берем расписание для понедельника (см perl TimeTargeting.pm:613)
                hoursCoef = getWeekdayCoefs().get(WeekdayType.MONDAY);
            }
        }
        return hoursCoef;
    }

    /**
     * По заданной дате возвращает действительный день недели в интервале MONDAY-SUNDAY
     */
    private WeekdayType getRealWeekdayType(LocalDate date) {
        return WeekdayType.getById(date.getDayOfWeek().getValue());
    }

    /**
     * Возвращает коэффициент (множитель) временного таргетинга.
     *
     * @param dateTime           время, для которого необходимо расчитать коэффициент
     * @param productionCalendar реализация производственного календаря {@link ProductionCalendar}, который знает,
     *                           какие будние дни праздничные, а какие выходные рабочие
     */
    public double calcTimeTargetCoef(LocalDateTime dateTime, ProductionCalendar productionCalendar) {
        HoursCoef hoursCoef = getHoursCoef(dateTime.toLocalDate(), productionCalendar);
        int hour = dateTime.getHour();

        return hoursCoef != null ? hoursCoef.getCoefForHour(hour) / 100. : PredefinedCoefs.ZERO.getDouble();
    }

    /**
     * @return строчку для сохранения настроек временного таргетинга
     */
    @Nonnull
    public String toRawFormat() {
        StringBuilder sb = new StringBuilder();
        ArrayList<WeekdayType> weekdayTypes = new ArrayList<>(weekdayCoefs.keySet());
        weekdayTypes.sort(Comparator.comparing(WeekdayType::getInternalNum));

        for (WeekdayType weekdayType : weekdayTypes) {
            sb.append(weekdayType.getInternalNum());
            HoursCoef hoursCoef = weekdayCoefs.get(weekdayType);
            for (int i = 0; i < 24; i++) {
                int coefForHour = hoursCoef.getCoefForHour(i);
                if (coefForHour > 0) {
                    sb.append((char) ('A' + i));
                    if (coefForHour != 100) {
                        sb.append((char) ('a' + coefForHour / 10));
                    }
                }
            }
        }

        if (preset != null) {
            sb.append(PROP_DELIMITER);
            sb.append("p:");
            switch (preset) {
                case ALL:
                    sb.append('a');
                    break;
                case WORKTIME:
                    sb.append('w');
                    break;
                case OTHER:
                    sb.append('o');
                    break;
                default:
                    throw new IllegalStateException("Unexpected value for Preset enum: " + preset);
            }
        }

        return sb.toString();
    }

    /**
     * Проверить, равны ли часы показа (без учёта коэффицентов и пресета)
     */
    public boolean equalsShowHours(TimeTarget that) {
        if (this == that) {
            return true;
        } else if (that == null) {
            return false;
        } else {
            for (var dayType : WeekdayType.values()) {
                var hours = weekdayCoefs.get(dayType);
                var thatHours = that.weekdayCoefs.get(dayType);
                for (int i = 0; i < 24; i++) {
                    var coef = hours == null || hours.getCoefForHour(i) == 0 ? 0 : 1;
                    var thatCoef = thatHours == null || thatHours.getCoefForHour(i) == 0 ? 0 : 1;
                    if (coef != thatCoef) {
                        return false;
                    }
                }
            }
            return true;
        }
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        TimeTarget that = (TimeTarget) o;
        return Objects.equals(weekdayCoefs, that.weekdayCoefs) &&
                preset == that.preset;
    }

    @Override
    public int hashCode() {
        return Objects.hash(weekdayCoefs, preset);
    }

    @Override
    public String toString() {
        return "TimeTarget{" +
                "preset=" + preset +
                ", weekdayCoefs=" + weekdayCoefs +
                '}';
    }

    public enum Preset {
        ALL,
        WORKTIME,
        OTHER
    }

    public enum PredefinedCoefs {
        ZERO(0),
        USUAL(100);

        private int coef;

        PredefinedCoefs(int coef) {
            this.coef = coef;
        }

        public int getValue() {
            return coef;
        }

        public double getDouble() {
            return Double.valueOf(coef);
        }
    }
}

