package ru.yandex.direct.tracing;

import java.util.function.Supplier;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import ru.yandex.direct.tracing.data.TraceData;

/**
 * Object used for capturing metrics during execution
 */
public interface Trace {

    boolean FULL = false;
    boolean PARTIAL = true;

    /**
     * Returns trace id of this traced call
     */
    long getTraceId();

    /**
     * Returns parent id of this traced call
     */
    long getParentId();

    /**
     * Returns span id of this traced call
     */
    long getSpanId();

    /**
     * Returns service name for this traced call
     */
    String getService();

    /**
     * Returns method name for this traced call
     */
    String getMethod();

    /**
     * Returns tags for this traced call
     */
    String getTags();

    /**
     * Returns samplerate for this traced call
     * <p>
     * The probability of this trace being sampled is 1/samplerate.
     * The default value of 0 is used for determining default samplerate at runtime.
     */
    int getSamplerate();

    /**
     * Returns tll for this traced call
     * <p>
     * Traces are forced to be sampled up to ttl calls deep, including this traced call.
     */
    int getTtl();

    boolean getSkipInterception();

    double getCpuTime();

    /**
     * Overrides trace id of this traced call
     */
    void setTraceId(long traceId);

    /**
     * Overrides parent id of this traced call
     */
    void setParentId(long parentId);

    /**
     * Overrides span id of this traced call
     */
    void setSpanId(long spanId);

    /**
     * Overrides service name for this traced call
     */
    void setService(String service);

    /**
     * Overrides method name for this traced call
     */
    void setMethod(String method);

    /**
     * Overrides tags for this traced call
     */
    void setTags(String tags);

    /**
     * Overrides samplerate for this traced call
     * <p>
     * The probability of this trace being sampled is 1/samplerate.
     * The default value of 0 is used for determining default samplerate at runtime.
     *
     * @param samplerate samplerate for this traced call, greater than or equal to zero
     */
    void setSamplerate(int samplerate);

    /**
     * Overrides ttl for this traced call
     * <p>
     * Traces are forced to be sampled up to ttl calls deep, including this traced call.
     *
     * @param ttl ttl for this traced call, greater than or equal to zero
     */
    void setTtl(int ttl);

    void setSkipInterception(boolean skipInterception);

    /**
     * Returns elapsed time in seconds since the start of this traced call
     */
    double elapsed();

    /**
     * Starts profiling of a function call
     * <p>
     * The calling thread must call close on the returned object at the end of the function call.
     *
     * @param func     name of the function
     * @param tags     additional tags
     * @param objCount number of objects that the function is working on (for better aggregation)
     * @return function call tracking object
     */
    TraceProfile profile(String func, String tags, long objCount);

    /**
     * Starts profiling of a function call
     * <p>
     * The calling thread must call close on the returned object at the end of the function call.
     *
     * @param func name of the function
     * @param tags additional tags
     * @return function call tracking object
     */
    default TraceProfile profile(String func, String tags) {
        return profile(func, tags, 0);
    }

    /**
     * Starts profiling of a function call
     * <p>
     * The calling thread must call close on the returned object at the end of the function call.
     *
     * @param func name of the function
     * @return function call tracking object
     */
    default TraceProfile profile(String func) {
        return profile(func, "");
    }

    /**
     * Starts tracking of a remote method call
     * <p>
     * Any thread may call close on the returned object at the end of the remote method call.
     *
     * @param service name of the remote service
     * @param method  name of the remote method
     * @return service call tracking object
     */
    TraceChild child(String service, String method);

    /**
     * Stores a message at the current point in time
     *
     * @param message a message describing what happened at the current point in time
     */
    void mark(String message);

    /**
     * Stores an annotation for this traced call
     *
     * @param key   an annotation key
     * @param value a value for associating with the current key
     */
    void annotate(String key, String value);

    /**
     * Returns a snapshot of data captures by this traced call
     * <p>
     * The return value is null if there's no data to return, it normally happens if the trace has not been sampled.
     *
     * @param partial true for partial data, normally used by a background logger
     * @return TraceData or null
     */
    @Nullable
    TraceData snapshot(boolean partial);

    /**
     * Stops capturing data, making the next call to snapshot return null
     */
    void close();

    /**
     * This method is called when trace object is activated in the current thread
     */
    default void activate() {
        // nothing
    }

    /**
     * This method is called when trace object is deactivated in the current thread
     */
    default void deactivate() {
        // nothing
    }

    /**
     * Returns active trace for the current thread or a dummy object if there is no active trace at the moment.
     */
    @Nonnull
    static Trace current() {
        Trace trace = TraceHolder.currentTrace();
        return trace != null ? trace : DummyTrace.instance();
    }

    /**
     * Returns active trace for the current thread or a null if there is no active trace at the moment.
     */
    @Nullable
    static Trace currentOrNull() {
        return TraceHolder.currentTrace();
    }

    /**
     * Pushes and activates trace on top of the stack for the current thread
     */
    static void push(Trace trace) {
        TraceHolder.pushTrace(trace);
        TraceMdcAdapter.updateMDC(trace);
    }

    /**
     * Обновляет Mdc, связанный с текущим трейсом.
     * Требуется, поскольку {@link Trace} - мутабелен,
     * и как следствие - текущий трейс можно поменять используя ссылку - в обход {@link #push(Trace)}.
     */
    static void updateMdc() {
        TraceMdcAdapter.updateMDC(current());
    }

    /**
     * Приостаналивает выполнение текущего треда с профилированием времени сна
     *
     * @param millis длительность времени для сна в миллисекундах
     * @throws InterruptedException при прерывании выполнения текущего треда любым другим
     */
    static void sleep(long millis) throws InterruptedException {
        try (TraceProfile profile = current().profile("sleep")) {
            Thread.sleep(millis);
        }
    }

    /**
     * Pops and deactivates trace on top of the stack for the current thread
     * <p>
     * Returns null if there is no active trace at the moment.
     */
    @Nullable
    static Trace pop() {
        Trace result = TraceHolder.popTrace();
        TraceMdcAdapter.updateMDC(current());
        return result;
    }

    /**
     * Wraps a runnable making this trace current around the execution
     */
    default Runnable wrap(Runnable runnable) {
        return () -> {
            push(this);
            try {
                runnable.run();
            } finally {
                pop();
            }
        };
    }

    /**
     * Wraps a supplier making this trace current around the execution.
     */
    default <V> Supplier<V> wrap(Supplier<V> callable) {
        return () -> {
            push(this);
            try {
                return callable.get();
            } finally {
                pop();
            }
        };
    }
}
