package ru.yandex.direct.tracing;

import java.util.Collection;
import java.util.concurrent.Future;

import ru.yandex.direct.tracing.real.RealTrace;
import ru.yandex.direct.tracing.util.TraceUtil;

/**
 * Упрощение трейсинга
 * <pre>
 *     try(TraceGuard guard = traceHelper(method, tags)) {
 *         ... do something ...
 *     }
 * </pre>
 */
public class TraceHelper {
    private final String service;
    private final TraceLogger traceLogger;
    private final TraceInterceptor traceInterceptor;

    public TraceHelper(
            String service,
            TraceLogger traceLogger
    ) {
        this(service, traceLogger, null);
    }

    public TraceHelper(
            String service,
            TraceLogger traceLogger,
            TraceInterceptor traceInterceptor
    ) {
        this.service = service;
        this.traceLogger = traceLogger;
        this.traceInterceptor = traceInterceptor;
    }

    /**
     * Начало трейсинга - создаём объект-защитник, регистриуем
     * задачу по регулярному сбросу состояния
     *
     * @param trace
     * @return
     */
    public TraceGuard guard(Trace trace) {
        Trace.push(trace);
        Future registration;
        try {
            registration = traceLogger.register(trace);
        } catch (Exception e) {
            Trace.pop();
            traceLogger.dump(trace);
            throw e;
        }
        return new TraceGuard(traceLogger, trace, registration);
    }

    public TraceGuard guard(String method) {
        return guard(method, "");
    }

    public TraceGuard guard(String method, Collection<String> tags) {
        return guard(method, String.join(",", tags));
    }

    public TraceGuard guard(String method, String tags) {
        Trace trace = RealTrace.builder()
                .withInfo(service, method, tags)
                .withTraceInterceptor(traceInterceptor)
                .build();
        return guard(trace);
    }

    public TraceGuard guard(String method, String tags, long traceId, long spanId) {
        Trace trace = RealTrace.builder()
                .withTraceId(traceId)
                .withSpanId(spanId)
                .withInfo(service, method, tags)
                .withTraceInterceptor(traceInterceptor)
                .build();
        return guard(trace);
    }

    public TraceGuard guardFromHeader(String header, String method) {
        RealTrace.Builder builder = RealTrace.builder().withTraceInterceptor(traceInterceptor);
        RealTrace trace = TraceUtil.traceFromHeader(header, service, method, null, builder);
        return guard(trace);
    }

    /**
     * Получить название текущено сервиса (обычно полученное из конфига, tracing.service_name)
     *
     * @return
     */
    public String getService() {
        return service;
    }
}
