package ru.yandex.direct.tracing;

import javax.annotation.Nullable;

/**
 * Used for per-thread tracking of the trace stack
 */
abstract class TraceHolder {
    private static final ThreadLocal<Entry> THREAD_LOCAL = new ThreadLocal<>();

    private static class Entry {
        private final Entry next;
        private final Trace trace;

        Entry(Entry next, Trace trace) {
            this.next = next;
            this.trace = trace;
        }
    }

    private TraceHolder() {
        // deny instantiation
    }

    /**
     * Pushes and activates trace on top of the stack for the current thread
     */
    static void pushTrace(Trace trace) {
        if (trace == null) {
            throw new NullPointerException();
        }
        trace.activate();
        THREAD_LOCAL.set(new Entry(THREAD_LOCAL.get(), trace));
    }

    /**
     * Returns active trace object for the current thread
     */
    @Nullable
    static Trace currentTrace() {
        Entry entry = THREAD_LOCAL.get();
        return entry != null ? entry.trace : null;
    }

    /**
     * Pops and deactivates trace from the stack for the current thread
     */
    @Nullable
    static Trace popTrace() {
        Entry entry = THREAD_LOCAL.get();
        if (entry != null) {
            if (entry.next != null) {
                THREAD_LOCAL.set(entry.next);
            } else {
                THREAD_LOCAL.remove();
            }
            entry.trace.deactivate();
            return entry.trace;
        }
        return null;
    }
}
