package ru.yandex.direct.tracing;

import java.io.IOException;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.tracing.data.TraceData;
import ru.yandex.direct.tracing.util.TracePrettyFormatter;
import ru.yandex.direct.tracing.util.TraceTaskScheduler;

/**
 * Used for logging trace data
 */
@ParametersAreNonnullByDefault
public class TraceLogger {
    /**
     * Used for cancelling the execution of the current task
     */
    public class Cancelled extends RuntimeException {
    }

    private static final Logger logger = LoggerFactory.getLogger(TraceLogger.class);
    private static final Logger TRACE_LOGGER = LoggerFactory.getLogger("TRACE.log");
    private static final Logger TRACE_PRETTY_LOGGER = LoggerFactory.getLogger("TRACE.pretty");

    private TraceTaskScheduler scheduler;

    private final long period;
    private final TimeUnit unit;

    private final TracePrettyFormatter prettyFormatter;

    /**
     * Creates a default single-threaded scheduler for logging partial traces every second
     */
    public TraceLogger() {
        this(new TraceTaskScheduler.Default());
    }

    /**
     * Uses the specified scheduler for logging partial traces every 30 seconds
     */
    public TraceLogger(TraceTaskScheduler scheduler) {
        this(scheduler, 30, TimeUnit.SECONDS);
    }

    /**
     * Uses the specified scheduler for logging partial traces every period
     */
    public TraceLogger(TraceTaskScheduler scheduler, long period, TimeUnit unit) {
        this(scheduler, period, unit, TracePrettyFormatter.defaultFormatter());
    }

    /**
     * Uses the specified scheduler for logging partial traces every period
     */
    public TraceLogger(TraceTaskScheduler scheduler, long period, TimeUnit unit, TracePrettyFormatter prettyFormatter) {
        this.scheduler = scheduler;
        this.period = period;
        this.unit = unit;
        this.prettyFormatter = prettyFormatter;
    }

    /**
     * Snapshots and logs data for the trace
     *
     * @param trace   trace object
     * @param partial whether to log partial data
     * @return true if there was any data, false otherwise
     */
    public boolean dump(Trace trace, boolean partial) {
        TraceData data = trace.snapshot(partial);
        if (data != null) {
            try {
                TRACE_LOGGER.info(data.toJson());
            } catch (IOException e) {
                logger.error("JSON convertion failure", e);
            }
            if (TRACE_PRETTY_LOGGER.isInfoEnabled() && prettyFormatter != null) {
                try {
                    TRACE_PRETTY_LOGGER.info(prettyFormatter.format(data));
                } catch (RuntimeException e) {
                    logger.error("Error occured while pretty formatting", e);
                }
            }
            return true;
        }
        return false;
    }

    /**
     * Snapshots and logs data for the trace
     *
     * @param trace trace object
     * @return true if there was any data, false otherwise
     */
    public boolean dump(Trace trace) {
        return dump(trace, false);
    }

    /**
     * Registers trace object for background logging
     *
     * @param trace trace object
     * @return a Future object that may be used to cancel background logging
     */
    public Future register(final Trace trace) {
        return scheduler.schedulePeriodicTask(
                () -> {
                    logger.trace("Periodic dump: {}", trace);
                    if (!dump(trace, true)) {
                        throw new Cancelled();
                    }
                },
                period,
                unit);
    }
}
