package ru.yandex.direct.tracing;

import java.util.Random;

/**
 * Used for making decisions on whether to sample a trace
 */
public interface TraceSampler {
    /**
     * Returns the default samplerate for the specified trace object
     *
     * @param trace trace object
     * @return sample rate that is greater than or equal to 1
     */
    int defaultSampleRate(Trace trace);

    /**
     * Returns whether to sample the specified trace object
     *
     * @param trace trace object
     * @return true if the trace should be sampled and logged
     */
    boolean sample(Trace trace);

    /**
     * This sampler assigns 100% samplerate by default and uses random sampling otherwise
     */
    class Default implements TraceSampler {
        private static final Random RANDOM = new Random();
        private static final Default INSTANCE = new Default();

        @Override
        public int defaultSampleRate(Trace trace) {
            return 1;
        }

        @Override
        public boolean sample(Trace trace) {
            int samplerate = trace.getSamplerate();
            if (samplerate <= 1) {
                return true;
            }
            if (trace.getTtl() > 0) {
                return true;
            }
            int value = RANDOM.nextInt(samplerate);
            return value == 0;
        }

        public static Default instance() {
            return INSTANCE;
        }
    }
}
