package ru.yandex.direct.tracing.data;

import java.util.Iterator;
import java.util.Objects;
import java.util.Optional;
import java.util.OptionalLong;

import javax.annotation.Nullable;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonGetter;
import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.google.common.base.CharMatcher;

import static java.util.Arrays.asList;

/**
 * Директ добавляет в sql запросы мета-информацию из трейсинга: идентификатор http-запроса,
 * сервис (API, Веб-интерфейс и т.д.) и метод. Этот класс позволяет её извлечь и прочитать.
 * <p>
 * Код, который эту инфу добавляет:
 * в perl: https://svn.yandex-team.ru/websvn/wsvn/direct-utils/yandex-lib/dbtools/lib/Yandex/DBTools
 * .pm?op=blame&rev=9732#l912
 * в java: https://a.yandex-team.ru/arc/trunk/arcadia/direct/common/src/main/java/ru/yandex/direct/common/db/wrapper
 * /DatabaseWrapper.java?rev=2718962#L237-248
 */
@SuppressWarnings("OptionalUsedAsFieldOrParameterType")
public class DirectTraceInfo {
    private static final DirectTraceInfo EMPTY = new DirectTraceInfo(
            OptionalLong.empty(), Optional.empty(), Optional.empty(), OptionalLong.empty(), Optional.empty(),
            false);

    private static final CharMatcher LETTER_SPACE_MATCHER = CharMatcher.inRange('a', 'z')
            .or(CharMatcher.inRange('A', 'Z'))
            .or(CharMatcher.whitespace())
            .precomputed();
    private static final String COMMENT_START = "/*";
    private static final String COMMENT_END = "*/";
    private static final String OPERATOR_PREFIX = "operator=";
    private static final String ESS_TAG_PREFIX = "ess=";
    private static final String RESHARDING_PREFIX = "resharding=";
    private static final int MAX_COMMENT_LENGTH = 1_000;

    private final OptionalLong reqId;
    private final Optional<String> service;
    private final Optional<String> method;
    private final OptionalLong operatorUid;
    private final Optional<String> essTag;
    private final boolean resharding;

    public DirectTraceInfo(OptionalLong reqId, Optional<String> service, Optional<String> method,
                           OptionalLong operatorUid, Optional<String> essTag, boolean resharding) {
        this.reqId = reqId;
        this.service = service;
        this.method = method;
        this.operatorUid = operatorUid;
        this.essTag = essTag;
        this.resharding = resharding;
    }

    @JsonCreator
    public DirectTraceInfo(@JsonProperty("req_id") @Nullable Long reqId,
                           @JsonProperty("service") @Nullable String service,
                           @JsonProperty("method") @Nullable String method,
                           @JsonProperty("operator_uid") @Nullable Long operatorUid,
                           @JsonProperty("ess_tag") @Nullable String essTag,
                           @JsonProperty("resharding") boolean resharding) {
        this.reqId = reqId == null ? OptionalLong.empty() : OptionalLong.of(reqId);
        this.service = Optional.ofNullable(service);
        this.method = Optional.ofNullable(method);
        this.operatorUid = operatorUid == null ? OptionalLong.empty() : OptionalLong.of(operatorUid);
        this.essTag = Optional.ofNullable(essTag);
        this.resharding = resharding;
    }

    public DirectTraceInfo(OptionalLong reqId, Optional<String> service, Optional<String> method) {
        this(reqId, service, method, OptionalLong.empty(), Optional.empty(), false);
    }

    public DirectTraceInfo(OptionalLong reqId, Optional<String> service, Optional<String> method,
                           OptionalLong operatorUid) {
        this(reqId, service, method, operatorUid, Optional.empty(), false);
    }

    public DirectTraceInfo(long reqId) {
        this(OptionalLong.of(reqId), Optional.empty(), Optional.empty(), OptionalLong.empty(), Optional.empty(), false);
    }

    public static DirectTraceInfo empty() {
        return EMPTY;
    }

    /**
     * Парсит комментарий. Если не удалось корректно распарсить, то возвращает {@link DirectTraceInfo},
     * собранный из того, что удалось корректно распарсить.
     */
    public static DirectTraceInfo extractIgnoringErrors(String query) {
        try {
            return extract(query);
        } catch (ParseException e) {
            return e.getFallback();
        }
    }

    /**
     * Парсит комментарий. Если не удалось корректно распарсить, то бросает {@link ParseException}, в которой есть
     * {@link DirectTraceInfo} с тем, что удалось корректно распарсить.
     */
    public static DirectTraceInfo extract(String query) {
        int start = query.indexOf(COMMENT_START);
        if (start == -1) {
            return EMPTY;
        }
        if (start > 0 && !LETTER_SPACE_MATCHER.matchesAllOf(query.substring(0, start))) {
            return EMPTY;
        }

        int end = query.indexOf(COMMENT_END, start + COMMENT_START.length());
        if (end == -1 || end <= start || end - start > MAX_COMMENT_LENGTH) {
            return EMPTY;
        }

        String comment = query.substring(start + COMMENT_START.length(), end).trim();
        return parseSqlComment(comment);
    }

    private static DirectTraceInfo parseSqlComment(String comment) {
        // Схема комментария - это список значений, разделённых `:`
        // Первое значение - строка "reqid", обязательная.
        // Второе значение - число span id (здесь он называется req id), обязательное.
        // Все следующие значения необязательные.
        // Третье значение - сервис, строка.
        // Четвёртое значение - метод, строка.
        // Далее идут строки `ключ=значение`.

        String[] parts = comment.split(":");
        if (parts.length < 2) {
            return EMPTY;
        }
        Iterator<String> iter = asList(parts).iterator();
        if (!iter.next().equals("reqid")) {
            return EMPTY;
        }

        long reqid;
        try {
            reqid = Long.parseLong(iter.next());
        } catch (NumberFormatException ex) {
            return EMPTY;
        }

        Optional<String> service = iter.hasNext() ? Optional.of(iter.next()) : Optional.empty();
        Optional<String> method = iter.hasNext() ? Optional.of(iter.next()) : Optional.empty();

        OptionalLong operator = OptionalLong.empty();
        Optional<String> essTag = Optional.empty();
        boolean resharding = false;
        String error = null;
        while (iter.hasNext()) {
            String part = iter.next();
            if (part.startsWith(OPERATOR_PREFIX)) {
                try {
                    operator = OptionalLong.of(Long.parseLong(part.substring(OPERATOR_PREFIX.length())));
                } catch (NumberFormatException ex) {
                    error = String.format("Bad operator uid in %s", comment);
                }
            }
            if (part.startsWith(ESS_TAG_PREFIX)) {
                essTag = Optional.of(part.substring(ESS_TAG_PREFIX.length()));
            }
            if (part.startsWith(RESHARDING_PREFIX) && !part.substring(RESHARDING_PREFIX.length()).equals("0")) {
                resharding = true;
            }
        }

        DirectTraceInfo result = new DirectTraceInfo(OptionalLong.of(reqid), service, method, operator, essTag,
                resharding);
        if (error == null) {
            return result;
        } else {
            throw new ParseException(error, result);
        }
    }

    @JsonGetter("req_id")
    public Long getUnboxedReqId() {
        return reqId.isPresent() ? reqId.getAsLong() : null;
    }

    @JsonGetter("service")
    public String getUnboxedService() {
        return service.orElse(null);
    }

    @JsonGetter("method")
    public String getUnboxedMethod() {
        return method.orElse(null);
    }

    @JsonGetter("operator_uid")
    public Long getUnboxedOperatorUid() {
        return operatorUid.isPresent() ? operatorUid.getAsLong() : null;
    }

    @JsonGetter("ess_tag")
    public String getUnboxedEssTag() {
        return essTag.orElse(null);
    }

    @JsonGetter("resharding")
    public boolean getResharding() {
        return resharding;
    }

    @JsonIgnore
    public OptionalLong getReqId() {
        return reqId;
    }

    @JsonIgnore
    public Optional<String> getService() {
        return service;
    }

    @JsonIgnore
    public Optional<String> getMethod() {
        return method;
    }

    @JsonIgnore
    public OptionalLong getOperatorUid() {
        return operatorUid;
    }

    @JsonIgnore
    public Optional<String> getEssTag() {
        return essTag;
    }

    @Override
    public String toString() {
        return "DirectTraceInfo{" +
                "reqId=" + reqId +
                ", service=" + service +
                ", method=" + method +
                ", operatorUid=" + operatorUid +
                ", essTag=" + essTag +
                ", resharding=" + resharding +
                '}';
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        DirectTraceInfo that = (DirectTraceInfo) o;
        return Objects.equals(reqId, that.reqId) &&
                Objects.equals(service, that.service) &&
                Objects.equals(method, that.method) &&
                Objects.equals(operatorUid, that.operatorUid) &&
                Objects.equals(essTag, that.essTag) &&
                Objects.equals(resharding, that.resharding);
    }

    @Override
    public int hashCode() {
        return Objects.hash(reqId, service, method, operatorUid, essTag, resharding);
    }

    public static class ParseException extends RuntimeException {
        private final DirectTraceInfo fallback;

        ParseException(String message, DirectTraceInfo fallback) {
            super(message);
            this.fallback = fallback;
        }

        /**
         * Объект, собранный из того, что удалось распарсить
         */
        public DirectTraceInfo getFallback() {
            return fallback;
        }
    }
}
