package ru.yandex.direct.tracing.data;

import java.io.IOException;
import java.util.List;

import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonToken;
import com.fasterxml.jackson.core.TreeNode;

import static com.google.common.base.Preconditions.checkArgument;

/**
 * Сборник шорткатов для написания кастомных весериализаторов
 */
public class JsonParserHelper {
    private final JsonParser parser;

    public JsonParserHelper(JsonParser parser) {
        this.parser = parser;
    }

    /**
     * Проверить, что мы действительно читаем массив, прочитать массив объектов типа clazz, добавить их в list.
     */
    public <T> void readToList(List<T> list, Class<T> clazz) throws IOException {
        checkArgument(parser.currentToken() == JsonToken.START_ARRAY);
        while (parser.nextToken() != JsonToken.END_ARRAY) {
            list.add(parser.readValueAs(clazz));
        }
        checkArgument(parser.currentToken() == JsonToken.END_ARRAY);
    }

    public long nextLong() throws IOException {
        parser.nextToken();
        return parser.getValueAsLong();
    }

    public int nextInt() throws IOException {
        parser.nextToken();
        return parser.getValueAsInt();
    }

    public boolean nextBoolean() throws IOException {
        parser.nextToken();
        return parser.getValueAsBoolean();
    }

    public double nextDouble() throws IOException {
        parser.nextToken();
        return parser.getValueAsDouble();
    }

    public String nextString() throws IOException {
        parser.nextToken();
        return parser.getValueAsString();
    }

    /**
     * Специфика TraceData - объекты сериализуются в массивы.
     * Функция позволяет написать потребитель элементов массива, не думая о объёртке.
     */
    public <T> T readArray(IoSupplier<T> supplier) throws IOException {
        checkArgument(parser.currentToken() == JsonToken.START_ARRAY,
                "Expected start_array, current token {}", parser.currentToken());

        T ret = supplier.get();

        parser.nextToken();
        checkArgument(parser.currentToken() == JsonToken.END_ARRAY,
                "Expected end_array, current token {}", parser.currentToken());

        return ret;
    }

    public TreeNode nextTree() throws IOException {
        parser.nextToken();
        return parser.readValueAsTree();
    }

    @FunctionalInterface
    public interface IoSupplier<T> {
        T get() throws IOException;
    }
}
