package ru.yandex.direct.tracing.data;

import java.io.IOException;
import java.time.Instant;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.List;

import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.JsonToken;
import com.fasterxml.jackson.databind.DeserializationContext;
import com.fasterxml.jackson.databind.JsonDeserializer;
import com.fasterxml.jackson.databind.JsonSerializer;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializerProvider;
import com.fasterxml.jackson.databind.annotation.JsonSerialize;
import com.fasterxml.jackson.databind.module.SimpleModule;
import com.google.common.base.MoreObjects;

import static com.google.common.base.Preconditions.checkArgument;

/**
 * Stores trace data suitable for logging
 */
@JsonSerialize(using = TraceData.Serializer.class)
public class TraceData {
    private static final ObjectMapper OBJECT_MAPPER;

    static {
        SimpleModule module = new SimpleModule();
        module.addDeserializer(TraceData.class, new Deserializer());
        module.addDeserializer(TraceDataTimes.class, new TraceDataTimes.Deserializer());
        module.addDeserializer(TraceDataProfile.class, new TraceDataProfile.Deserializer());
        module.addDeserializer(TraceDataChild.class, new TraceDataChild.Deserializer());
        module.addDeserializer(TraceDataMark.class, new TraceDataMark.Deserializer());
        module.addDeserializer(TraceDataAnnotation.class, new TraceDataAnnotation.Deserializer());
        OBJECT_MAPPER = new ObjectMapper();
        OBJECT_MAPPER.registerModule(module);
    }

    private static final int FORMAT = 3;
    private static final DateTimeFormatter LOG_TIME_FORMATTER = DateTimeFormatter
            .ofPattern("yyyy-MM-dd HH:mm:ss.SSSSSS")
            .withZone(ZoneId.of("UTC"));

    private Instant logTime;
    private String host;
    private int pid;
    private String service;
    private String method;
    private String tags;
    private long traceId;
    private long parentId;
    private long spanId;
    private int chunkIndex;
    private boolean chunkFinal;
    private double allEla;
    private int samplerate;
    private TraceDataTimes times = new TraceDataTimes();
    private final List<TraceDataProfile> profiles = new ArrayList<>();
    private final List<TraceDataChild> children = new ArrayList<>();
    private final List<TraceDataMark> marks = new ArrayList<>();
    private final List<TraceDataAnnotation> annotations = new ArrayList<>();

    public Instant getLogTime() {
        return logTime;
    }

    public void setLogTime(Instant logTime) {
        this.logTime = logTime;
    }

    public String getHost() {
        return host;
    }

    public void setHost(String host) {
        this.host = host;
    }

    public int getPid() {
        return pid;
    }

    public void setPid(int pid) {
        this.pid = pid;
    }

    public String getService() {
        return service;
    }

    public void setService(String service) {
        this.service = service;
    }

    public String getMethod() {
        return method;
    }

    public void setMethod(String method) {
        this.method = method;
    }

    public String getTags() {
        return tags;
    }

    public void setTags(String tags) {
        this.tags = tags;
    }

    public long getTraceId() {
        return traceId;
    }

    public void setTraceId(long traceId) {
        this.traceId = traceId;
    }

    public long getParentId() {
        return parentId;
    }

    public void setParentId(long parentId) {
        this.parentId = parentId;
    }

    public long getSpanId() {
        return spanId;
    }

    public void setSpanId(long spanId) {
        this.spanId = spanId;
    }

    public int getChunkIndex() {
        return chunkIndex;
    }

    public void setChunkIndex(int chunkIndex) {
        this.chunkIndex = chunkIndex;
    }

    public boolean isChunkFinal() {
        return chunkFinal;
    }

    public void setChunkFinal(boolean chunkFinal) {
        this.chunkFinal = chunkFinal;
    }

    public double getAllEla() {
        return allEla;
    }

    public void setAllEla(double allEla) {
        this.allEla = allEla;
    }

    public int getSamplerate() {
        return samplerate;
    }

    public void setSamplerate(int samplerate) {
        this.samplerate = samplerate;
    }

    public TraceDataTimes getTimes() {
        return times;
    }

    public List<TraceDataProfile> getProfiles() {
        return profiles;
    }

    public List<TraceDataChild> getChildren() {
        return children;
    }

    public List<TraceDataMark> getMarks() {
        return marks;
    }

    public List<TraceDataAnnotation> getAnnotations() {
        return annotations;
    }

    public String toJson() throws IOException {
        return OBJECT_MAPPER.writeValueAsString(this);
    }

    public static TraceData fromJson(String str) throws IOException {
        return OBJECT_MAPPER.readValue(str, TraceData.class);
    }

    @Override
    public String toString() {
        return MoreObjects.toStringHelper(this)
                .add("logTime", logTime)
                .add("host", host)
                .add("pid", pid)
                .add("service", service)
                .add("method", method)
                .add("tags", tags)
                .add("traceId", traceId)
                .add("parentId", parentId)
                .add("spanId", spanId)
                .add("chunkIndex", chunkIndex)
                .add("chunkFinal", chunkFinal)
                .add("allEla", allEla)
                .add("samplerate", samplerate)
                .add("times", times)
                .add("profiles", profiles)
                .add("children", children)
                .add("marks", marks)
                .add("annotations", annotations)
                .toString();
    }

    public static class Serializer extends JsonSerializer<TraceData> {

        private <T> void writeOptionalField(String name, T value, JsonGenerator gen, SerializerProvider serializers)
                throws IOException {
            if (value != null) {
                serializers.defaultSerializeField(name, value, gen);
            }
        }

        @Override
        public void serialize(TraceData data, JsonGenerator gen, SerializerProvider serializers)
                throws IOException, JsonProcessingException {
            gen.writeStartArray();

            gen.writeNumber(FORMAT);
            gen.writeString(LOG_TIME_FORMATTER.format(data.getLogTime()));
            gen.writeString(data.getHost());
            gen.writeNumber(data.getPid());

            gen.writeString(data.getService());
            gen.writeString(data.getMethod());
            gen.writeString(data.getTags());

            gen.writeNumber(data.getTraceId());
            gen.writeNumber(data.getParentId());
            gen.writeNumber(data.getSpanId());
            gen.writeNumber(data.getChunkIndex());
            gen.writeBoolean(data.isChunkFinal());

            gen.writeNumber(data.getAllEla());
            gen.writeNumber(data.getSamplerate() > 0 ? data.getSamplerate() : 1);

            gen.writeStartObject();
            writeOptionalField("times", data.getTimes(), gen, serializers);
            writeOptionalField("profile", data.getProfiles(), gen, serializers);
            writeOptionalField("services", data.getChildren(), gen, serializers);
            writeOptionalField("marks", data.getMarks(), gen, serializers);
            writeOptionalField("annotations", data.getAnnotations(), gen, serializers);
            gen.writeEndObject();
            gen.writeEndArray();
        }
    }

    public static class Deserializer extends JsonDeserializer<TraceData> {
        @Override
        public TraceData deserialize(
                JsonParser p, DeserializationContext ctx
        ) throws IOException, JsonProcessingException {
            JsonParserHelper ph = new JsonParserHelper(p);

            checkArgument(p.currentToken() == JsonToken.START_ARRAY);
            checkArgument(ph.nextInt() == FORMAT, "unsupported format");

            TraceData trace = new TraceData();

            trace.setLogTime(LOG_TIME_FORMATTER.parse(ph.nextString(), Instant::from));
            trace.setHost(ph.nextString());
            trace.setPid(ph.nextInt());

            trace.setService(ph.nextString());
            trace.setMethod(ph.nextString());
            trace.setTags(ph.nextString());

            trace.setTraceId(ph.nextLong());
            trace.setParentId(ph.nextLong());
            trace.setSpanId(ph.nextLong());
            trace.setChunkIndex(ph.nextInt());
            trace.setChunkFinal(ph.nextBoolean());

            trace.setAllEla(ph.nextDouble());
            trace.setSamplerate(ph.nextInt());

            checkArgument(p.nextToken() == JsonToken.START_OBJECT);
            while (p.nextToken() != JsonToken.END_OBJECT) {
                String name = p.getCurrentName();
                p.nextToken();
                switch (name) {
                    case "times":
                        trace.times = p.readValueAs(TraceDataTimes.class);
                        break;
                    case "profile":
                        ph.readToList(trace.profiles, TraceDataProfile.class);
                        break;
                    case "services":
                        ph.readToList(trace.children, TraceDataChild.class);
                        break;
                    case "marks":
                        ph.readToList(trace.marks, TraceDataMark.class);
                        break;
                    case "annotations":
                        ph.readToList(trace.annotations, TraceDataAnnotation.class);
                        break;
                    default:
                        // skip
                        p.readValueAsTree();
                }
            }

            p.nextToken();
            checkArgument(p.currentToken() == JsonToken.END_ARRAY, "current token {}", p.currentToken());

            return trace;
        }
    }
}

