package ru.yandex.direct.tracing.real;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import ru.yandex.direct.interruption.InterruptionChecker;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceChild;
import ru.yandex.direct.tracing.TraceInterceptor;
import ru.yandex.direct.tracing.TraceProfile;
import ru.yandex.direct.tracing.TraceSampler;
import ru.yandex.direct.tracing.data.TraceData;
import ru.yandex.direct.tracing.data.TraceDataAnnotation;
import ru.yandex.direct.tracing.data.TraceDataChild;
import ru.yandex.direct.tracing.data.TraceDataMark;
import ru.yandex.direct.tracing.util.HostnameUtils;
import ru.yandex.direct.tracing.util.ThreadUsedResources;
import ru.yandex.direct.tracing.util.ThreadUsedResourcesProvider;
import ru.yandex.direct.tracing.util.TraceClockProvider;
import ru.yandex.direct.tracing.util.TraceUtil;

import static java.util.stream.Collectors.toList;

/**
 * Real trace object that captures data
 */
public class RealTrace implements Trace {
    private final TraceClockProvider clock;

    private long traceId;
    private long parentId;
    private long spanId;
    private String service;
    private String method;
    private String tags;
    private int samplerate = 0;
    private int ttl = 0;
    private boolean skipInterception = false;

    private final long traceStart;
    private final RealTraceProfiler profiler;
    private final List<TraceDataChild> children = new ArrayList<>();
    private final List<TraceDataMark> marks = new ArrayList<>();
    private final List<TraceDataAnnotation> annotations = new ArrayList<>();

    private long chunkStart;
    private int chunkIndex = 1;
    private boolean sampleCalled = false;
    private TraceSampler sampler = TraceSampler.Default.instance();

    // Used to track cpu time in every active thread
    private final ThreadUsedResourcesProvider threadUsedResourcesProvider;
    private final ConcurrentHashMap<Thread, ThreadUsedResources> perThreadResources = new ConcurrentHashMap<>();
    private double usedCpuUserTime;
    private double usedCpuSystemTime;
    // в байтах
    private long allocatedBytes;

    /**
     * Support for child trace identifiers
     */
    public class Child implements TraceChild {
        private final long spanId;
        private final String service;
        private final String method;
        private final long callStart;
        private boolean closed;

        public Child(long spanId, String service, String method) {
            this.spanId = spanId;
            this.service = service;
            this.method = method;
            callStart = clock.nanoTime();
            closed = false;
        }

        @Override
        public long getTraceId() {
            return RealTrace.this.getTraceId();
        }

        @Override
        public long getParentId() {
            return RealTrace.this.getSpanId();
        }

        @Override
        public long getSpanId() {
            return spanId;
        }

        @Override
        public String getService() {
            return service;
        }

        @Override
        public String getMethod() {
            return method;
        }

        @Override
        public int getTtl() {
            int traceTtl = RealTrace.this.getTtl();
            return traceTtl > 0 ? traceTtl - 1 : 0;
        }

        @Override
        public double startedAt() {
            return TraceUtil.secondsFromNanoseconds(callStart - traceStart);
        }

        @Override
        public double elapsed() {
            return TraceUtil.secondsFromNanoseconds(clock.nanoTime() - callStart);
        }

        @Override
        public void close() {
            synchronized (RealTrace.this) {
                if (closed) {
                    throw new UnsupportedOperationException("cannot call close more than once");
                }
                children.add(new TraceDataChild(
                        getService(),
                        getMethod(),
                        getSpanId(),
                        startedAt(),
                        elapsed()));
                closed = true;
            }
        }
    }

    @SuppressWarnings("checkstyle:parameternumber")
    protected RealTrace(TraceClockProvider clock, long traceId, long parentId, long spanId, int ttl,
                        String service, String method, String tags,
                        ThreadUsedResourcesProvider threadUsedResourcesProvider,
                        TraceInterceptor traceInterceptor
    ) {
        this.clock = clock;
        this.traceId = traceId;
        this.parentId = parentId;
        this.spanId = spanId;
        this.ttl = ttl;
        this.service = service;
        this.method = method;
        this.tags = tags;
        this.traceStart = clock.nanoTime();
        this.profiler = new RealTraceProfiler(clock, traceInterceptor, service, method);
        this.chunkStart = traceStart;
        this.threadUsedResourcesProvider = threadUsedResourcesProvider;
    }

    @Override
    public synchronized long getTraceId() {
        return traceId;
    }

    @Override
    public synchronized void setTraceId(long traceId) {
        this.traceId = traceId;
    }

    @Override
    public synchronized long getParentId() {
        return parentId;
    }

    @Override
    public synchronized void setParentId(long parentId) {
        this.parentId = parentId;
    }

    @Override
    public synchronized long getSpanId() {
        return spanId;
    }

    @Override
    public synchronized void setSpanId(long spanId) {
        this.spanId = spanId;
    }

    @Override
    public synchronized String getService() {
        return service;
    }

    @Override
    public synchronized void setService(String service) {
        this.service = service;
        profiler.setService(service);
    }

    @Override
    public synchronized String getMethod() {
        return method;
    }

    @Override
    public synchronized void setMethod(String method) {
        this.method = method;
        profiler.setMethod(method);
    }

    @Override
    public synchronized String getTags() {
        return tags;
    }

    @Override
    public synchronized void setTags(String tags) {
        this.tags = tags;
    }

    @Override
    public synchronized int getSamplerate() {
        return samplerate;
    }

    @Override
    public synchronized void setSamplerate(int samplerate) {
        if (samplerate < 0) {
            throw new IllegalArgumentException("samplerate cannot be less than 0");
        }
        this.samplerate = samplerate;
    }

    @Override
    public synchronized int getTtl() {
        return ttl;
    }

    @Override
    public synchronized void setTtl(int ttl) {
        if (ttl < 0) {
            throw new IllegalArgumentException("ttl cannot be less than 0");
        }
        this.ttl = ttl;
    }

    @Override
    public synchronized boolean getSkipInterception() {
        return skipInterception;
    }

    @Override
    public synchronized void setSkipInterception(boolean skipInterception) {
        this.skipInterception = skipInterception;
    }

    @Override
    public double getCpuTime() {
        return usedCpuUserTime + usedCpuSystemTime;
    }

    @Override
    public double elapsed() {
        return TraceUtil.secondsFromNanoseconds(clock.nanoTime() - traceStart);
    }

    @Override
    public TraceProfile profile(String func, String tags, long objCount) {
        InterruptionChecker.checkInterruption();
        return profiler.profile(func, tags, objCount, skipInterception);
    }

    @Override
    public Child child(String service, String method) {
        InterruptionChecker.checkInterruption();
        return new Child(TraceUtil.randomId(), service, method);
    }

    @Override
    public synchronized void mark(String message) {
        if (message == null) {
            throw new NullPointerException();
        }
        if (marks != null) {
            marks.add(new TraceDataMark(elapsed(), message));
        }
    }

    @Override
    public synchronized void annotate(String key, String value) {
        if (key == null || value == null) {
            throw new NullPointerException();
        }
        if (annotations != null) {
            annotations.add(new TraceDataAnnotation(key, value));
        }
    }

    public synchronized TraceSampler getSampler() {
        return sampler;
    }

    public synchronized void setSampler(TraceSampler sampler) {
        if (sampler == null) {
            throw new NullPointerException();
        }
        if (sampleCalled) {
            throw new UnsupportedOperationException("sample has already been called");
        }
        this.sampler = sampler;
    }

    @Override
    public synchronized TraceData snapshot(boolean partial) {
        if (chunkIndex == 0) {
            return null;
        }
        if (samplerate < 1) {
            setSamplerate(sampler.defaultSampleRate(this));
        }
        if (!sampleCalled) {
            sampleCalled = true;
            if (!sampler.sample(this)) {
                close();
                return null;
            }
        }
        saveThreadCpuTime();
        long currentTime = clock.nanoTime();
        TraceData data = new TraceData();
        data.setLogTime(clock.instant());
        data.setHost(HostnameUtils.hostname());
        data.setPid(0); // cannot log pid
        data.setService(service);
        data.setMethod(method);
        data.setTags(tags);
        data.setTraceId(traceId);
        data.setParentId(parentId);
        data.setSpanId(spanId);
        data.setChunkIndex(chunkIndex);
        data.setChunkFinal(!partial);
        data.setAllEla(TraceUtil.secondsFromNanoseconds(currentTime - traceStart));
        data.setSamplerate(samplerate);
        data.getTimes().setEla(TraceUtil.secondsFromNanoseconds(currentTime - chunkStart));
        data.getTimes().setCpuUserTime(usedCpuUserTime);
        data.getTimes().setCpuSystemTime(usedCpuSystemTime);
        data.getTimes().setMem(allocatedBytes / 1048576.0);
        data.getChildren().addAll(children);
        data.getMarks().addAll(marks);
        data.getAnnotations().addAll(annotations);
        // сортируем, чтобы удобней было читать в логе
        data.getProfiles().addAll(
                profiler.snapshot(currentTime)
                        .stream()
                        .sorted((a, b) -> Double.compare(b.getAllEla(), a.getAllEla()))
                        .collect(toList())
        );
        if (partial) {
            // prepare for the next chunk
            chunkStart = currentTime;
            chunkIndex++;
            children.clear();
            marks.clear();
            annotations.clear();
            usedCpuUserTime = 0;
            usedCpuSystemTime = 0;
            allocatedBytes = 0;
        } else {
            close();
        }
        return data;
    }

    @Override
    public synchronized void close() {
        // causes the next snapshot call to return null
        chunkIndex = 0;
    }

    // сохраняет накопленные cpu-times из потредового хранилища в трейс
    private void saveThreadCpuTime() {
        for (Map.Entry<Thread, ThreadUsedResources> entry : perThreadResources.entrySet()) {
            ThreadUsedResources prev = entry.getValue();
            ThreadUsedResources now = threadUsedResourcesProvider.getThreadCpuTime(entry.getKey());
            if (now.isReliableCpuTimes() && prev.isReliableCpuTimes() && perThreadResources.replace(entry.getKey(),
                    prev, now)) {
                addThreadUsedResources(prev, now);
            }
        }
    }

    private synchronized void addThreadUsedResources(ThreadUsedResources prev, ThreadUsedResources now) {
        long userDelta = now.getUserTime() - prev.getUserTime();
        if (userDelta > 0) {
            usedCpuUserTime += TraceUtil.secondsFromNanoseconds(userDelta);
        }
        long systemDelta = now.getSystemTime() - prev.getSystemTime();
        if (systemDelta > 0) {
            usedCpuSystemTime += TraceUtil.secondsFromNanoseconds(systemDelta);
        }
        long memoryDelta = now.getAllocatedBytes() - prev.getAllocatedBytes();
        if (memoryDelta > 0) {
            allocatedBytes += memoryDelta;
        }
    }

    @Override
    public void activate() {
        profiler.activate();
        perThreadResources.put(
                Thread.currentThread(),
                threadUsedResourcesProvider.getCurrentThreadCpuTime()
        );
    }

    @Override
    public void deactivate() {
        ThreadUsedResources prev = perThreadResources.remove(Thread.currentThread());
        if (prev != null) {
            addThreadUsedResources(prev, threadUsedResourcesProvider.getCurrentThreadCpuTime());
        }
        profiler.deactivate();
    }

    public static Builder builder() {
        return new Builder();
    }

    public static class Builder {
        private TraceClockProvider clock;
        private ThreadUsedResourcesProvider threadUsedResourcesProvider;

        private long traceId = 0;
        private boolean traceIdSpecified = false;
        private long parentId = 0;
        private long spanId = 0;
        private int ttl = 0;
        private boolean spanIdSpecified = false;
        private String service;
        private String method;
        private String tags;
        private TraceInterceptor traceInterceptor;

        public Builder withClock(TraceClockProvider clock) {
            this.clock = clock;
            return this;
        }

        public Builder withTraceId(long traceId) {
            this.traceId = traceId;
            this.traceIdSpecified = true;
            return this;
        }

        public Builder withParentId(long parentId) {
            this.parentId = parentId;
            return this;
        }

        public Builder withSpanId(long spanId) {
            this.spanId = spanId;
            this.spanIdSpecified = true;
            return this;
        }

        public Builder withService(String service) {
            this.service = service;
            return this;
        }

        public Builder withMethod(String method) {
            this.method = method;
            return this;
        }

        public Builder withTags(String tags) {
            this.tags = tags;
            return this;
        }

        public Builder withThreadCpuTimeProvider(ThreadUsedResourcesProvider threadUsedResourcesProvider) {
            this.threadUsedResourcesProvider = threadUsedResourcesProvider;
            return this;
        }

        public Builder withTtl(int ttl) {
            this.ttl = ttl;
            return this;
        }

        public Builder withTraceInterceptor(TraceInterceptor traceInterceptor) {
            this.traceInterceptor = traceInterceptor;
            return this;
        }

        public Builder withIds(long traceId, long parentId, long spanId) {
            return this
                    .withTraceId(traceId)
                    .withParentId(parentId)
                    .withSpanId(spanId);
        }

        public Builder withInfo(String service, String method, String tags) {
            return this
                    .withService(service)
                    .withMethod(method)
                    .withTags(tags);
        }

        public RealTrace build() {
            long generatedId = !traceIdSpecified && !spanIdSpecified ? TraceUtil.randomId() : 0;
            return new RealTrace(
                    clock != null ? clock : TraceClockProvider.Default.instance(),
                    traceIdSpecified ? traceId : spanIdSpecified ? spanId : generatedId,
                    parentId,
                    spanIdSpecified ? spanId : traceIdSpecified ? traceId : generatedId,
                    ttl,
                    service, method, tags,
                    threadUsedResourcesProvider != null
                            ? threadUsedResourcesProvider
                            : ThreadUsedResourcesProvider.instance(),
                    traceInterceptor
            );
        }
    }
}
