package ru.yandex.direct.tracing.util;

import java.lang.management.ManagementFactory;
import java.lang.management.ThreadMXBean;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Provides cpu time for any thread
 */
public class ThreadUsedResourcesProvider {
    private static final Logger logger = LoggerFactory.getLogger(ThreadUsedResourcesProvider.class);
    private static final ThreadUsedResourcesProvider INSTANCE = new ThreadUsedResourcesProvider();

    private static final ThreadMXBean THREAD_MX_BEAN = ManagementFactory.getThreadMXBean();

    public ThreadUsedResources getCurrentThreadCpuTime() {
        long allocated = getCurrentThreadAllocatedBytes();
        long userTime = -1;
        long cpuTime = -1;
        try {
            userTime = THREAD_MX_BEAN.getCurrentThreadUserTime();
            cpuTime = THREAD_MX_BEAN.getCurrentThreadCpuTime();
        } catch (UnsupportedOperationException e) {
            logger.debug("Can't get cpu time", e);
        }
        return new ThreadUsedResources(userTime, cpuTime, allocated);
    }

    public ThreadUsedResources getThreadCpuTime(Thread thread) {
        long allocated = getThreadAllocatedBytes(thread);
        long userTime = -1;
        long cpuTime = -1;
        try {
            userTime = THREAD_MX_BEAN.getThreadUserTime(thread.getId());
            cpuTime = THREAD_MX_BEAN.getThreadCpuTime(thread.getId());
        } catch (UnsupportedOperationException e) {
            logger.debug("Can't get cpu time", e);
        }
        return new ThreadUsedResources(userTime, cpuTime, allocated);
    }

    protected long getCurrentThreadAllocatedBytes() {
        return getThreadAllocatedBytes(Thread.currentThread());
    }

    protected long getThreadAllocatedBytes(Thread thread) {
        try {
            if (THREAD_MX_BEAN instanceof com.sun.management.ThreadMXBean) {
                com.sun.management.ThreadMXBean sunMxBean = (com.sun.management.ThreadMXBean) THREAD_MX_BEAN;
                long allocatedBytes = sunMxBean.getThreadAllocatedBytes(thread.getId());
                if (allocatedBytes >= 0) {
                    return allocatedBytes;
                } else {
                    return -1;
                }
            } else {
                return -1;
            }
        } catch (RuntimeException e) {
            logger.debug("Can't get allocated bytes info", e);
            return -1;
        }
    }

    public static ThreadUsedResourcesProvider instance() {
        return INSTANCE;
    }
}
