package ru.yandex.direct.tracing.util;

import javax.annotation.Nullable;
import javax.annotation.WillClose;

import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.real.RealTrace;

import static com.google.common.base.Preconditions.checkState;

/**
 * Простой класс, позволяющий быстро померить скорость выполнения кода в main(), если лень использовать jmh.
 * При вызове close() информация о потраченных ресурсах печатается в stdout.
 * <pre>
 * {@code
 *     public static void main(String[] args) {
 *         try (TraceBenchmark ignore = TraceBenchmark.started()) {
 *             code();
 *         }
 *     }
 * }
 * </pre>
 */
public class TraceBenchmark implements RuntimeCloseable {
    private static final TracePrettyFormatter FORMATTER = new TracePrettyFormatter(Integer.MAX_VALUE, true);
    private Trace trace;

    private TraceBenchmark(Trace trace) {
        this.trace = trace;
    }

    public static TraceBenchmark started() {
        return started(null);
    }

    /**
     * Создаёт объект с запущенным трейсом
     */
    public static TraceBenchmark started(@Nullable String name) {
        Trace trace = RealTrace.builder()
                .withService("benchmark")
                .withMethod(name)
                .build();
        Trace.push(trace);
        return new TraceBenchmark(trace);
    }

    public Trace trace() {
        checkState(trace != null);
        return trace;
    }

    @Override
    public void close() {
        stopAndPrint();
    }

    /**
     * Распечатать информацию о потрпченных ресурсах в stdout.
     */
    @WillClose
    public void stopAndPrint() {
        checkState(trace != null);
        Trace.pop();
        System.out.println(FORMATTER.format(trace.snapshot(false)));
        trace = null;
    }

}
