package ru.yandex.direct.tracing.util;

import java.util.Comparator;
import java.util.List;

import ru.yandex.direct.tracing.data.TraceData;
import ru.yandex.direct.tracing.data.TraceDataProfile;
import ru.yandex.direct.tracing.data.TraceDataTimes;

import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.tracing.util.TracePrettyFormatter.Color.BLUE;
import static ru.yandex.direct.tracing.util.TracePrettyFormatter.Color.GREEN;
import static ru.yandex.direct.tracing.util.TracePrettyFormatter.Color.RED;
import static ru.yandex.direct.tracing.util.TracePrettyFormatter.Color.RESET;

public class TracePrettyFormatter {
    private final int profileTopSize;
    private final boolean useAnsiColors;
    private final NumberColorizer elaColorizer;
    private final NumberColorizer memColorizer;

    public TracePrettyFormatter(int profileTopSize, boolean useAnsiColors) {
        this.profileTopSize = profileTopSize;
        this.useAnsiColors = useAnsiColors;
        if (useAnsiColors) {
            elaColorizer = new NumberColorizer(GREEN, new double[]{1}, new Color[]{RED});
            memColorizer = new NumberColorizer(GREEN, new double[]{100}, new Color[]{RED});
        } else {
            elaColorizer = NumberColorizer.empty();
            memColorizer = NumberColorizer.empty();
        }
    }

    public static TracePrettyFormatter defaultFormatter() {
        return new TracePrettyFormatter(5, true);
    }

    /**
     * Форматирование Trace-а для удобного чтения
     * (аналог format-trace-log)
     */
    public String format(TraceData data) {
        StringBuilder sb = new StringBuilder("\n");

        writeMethod(sb, data);
        sb.append('\t');

        writeTimes(sb, data);
        sb.append('\t');

        writeIds(sb, data);
        if (!data.isChunkFinal() || data.getChunkIndex() != 0) {
            sb.append('\t');
            writeChunkInfo(sb, data);
        }

        writeProfiles(sb, data);

        return sb.toString();
    }

    private void writeChunkInfo(StringBuilder sb, TraceData data) {
        sb.append("chunk:")
                .append(data.getChunkIndex())
                .append('/')
                .append(data.isChunkFinal() ? "FIN" : "?");
    }

    private void writeProfiles(StringBuilder sb, TraceData data) {
        if (data.getProfiles() != null) {
            List<TraceDataProfile> sortedProfiles = data.getProfiles()
                    .stream()
                    .sorted(Comparator.comparingDouble(TraceDataProfile::getAllEla).reversed())
                    .collect(toList());
            for (int i = 0; i < profileTopSize && i < sortedProfiles.size(); i++) {
                TraceDataProfile profile = sortedProfiles.get(i);
                writeProfileItem(sb, profile);
            }
            if (sortedProfiles.size() > profileTopSize) {
                sb.append(" ... // ").append(sortedProfiles.size() - profileTopSize).append(" items more");
            }
        }
    }

    private void writeProfileItem(StringBuilder sb, TraceDataProfile profile) {
        sb.append("\n    ")
                .append(profile.getFunc());
        if (profile.getTags() != null && !profile.getTags().isEmpty()) {
            sb.append('/').append(profile.getTags());
        }
        sb.append("\tela:")
                .append(formatEla(profile.getAllEla()));
        if (profile.getChildrenEla() > 0) {
            sb.append("(-").append(formatEla(profile.getChildrenEla())).append(')');
        }
        if (profile.getCalls() > 1) {
            sb.append("\tcalls:").append(profile.getCalls());
        }
        if (profile.getObjCount() > 0) {
            sb.append("\tobj:").append(profile.getObjCount());
        }
    }

    private void writeIds(StringBuilder sb, TraceData data) {
        sb.append(data.getTraceId());
        if (data.getTraceId() != data.getSpanId()) {
            sb.append('/')
                    .append(data.getParentId())
                    .append('/')
                    .append(data.getSpanId());
        }
    }

    private void writeTimes(StringBuilder sb, TraceData data) {
        TraceDataTimes times = data.getTimes();
        sb.append("ela:")
                .append(elaColorizer.colorize(formatEla(times.getEla()), times.getEla()))
                .append(",cpu:")
                .append(formatEla(times.getCpuUserTime()))
                .append('/')
                .append(formatEla(times.getCpuSystemTime()))
                .append(",mem:")
                .append(memColorizer.colorize(formatMem(times.getMem()), times.getMem()));
    }

    private void writeMethod(StringBuilder sb, TraceData data) {
        sb.append(data.getService())
                .append('/')
                .append(colorized(data.getMethod(), BLUE));
        if (data.getTags() != null && !data.getTags().isEmpty()) {
            sb.append('/').append(data.getTags());
        }
    }

    // человеко-любивое форматирование времени
    private static String formatEla(double ela) {
        if (ela >= 0.001 || ela < 1e-6) {
            return String.format("%.3f", ela);
        } else {
            return String.format("%.6f", ela);
        }
    }

    // человеко-любивое форматирование аллоцированной памяти
    private static String formatMem(double mem) {
        if (mem < 10) {
            return String.format("+%.3fm", mem);
        } else {
            return String.format("+%.0fm", mem);
        }
    }

    private String colorized(String text, Color color) {
        if (useAnsiColors) {
            return color.seq + text + RESET.seq;
        } else {
            return text;
        }
    }

    private static class NumberColorizer {
        private final Color defaultColor;
        private final double[] borders;
        private final Color[] colors;

        NumberColorizer(Color defaultColor, double[] borders, Color[] colors) {
            if (borders != null && (colors == null || borders.length != colors.length)) {
                throw new IllegalArgumentException("Invalid colors array");
            }
            this.defaultColor = defaultColor;
            this.borders = borders;
            this.colors = colors;
        }

        private String colorize(String text, double val) {
            Color color = defaultColor;
            if (borders != null) {
                for (int i = 0; i < borders.length; i++) {
                    if (val >= borders[i]) {
                        color = colors[i];
                    }
                }
            }
            if (color == null) {
                return text;
            } else {
                return color.seq + text + RESET.seq;
            }
        }

        public static NumberColorizer empty() {
            return new NumberColorizer(null, null, null);
        }
    }

    enum Color {
        RESET(0),
        RED(31),
        GREEN(32),
        YELLOW(33),
        BLUE(34),
        MAGENTA(35),
        CYAN(36),
        WHITE(37);

        final int code;
        final String seq;

        Color(int code) {
            this.code = code;
            this.seq = "" + (char) 27 + "[" + code + "m";
        }
    }
}
