package ru.yandex.direct.tracing.util;

import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;

import com.google.common.util.concurrent.ThreadFactoryBuilder;

/**
 * Interface is used for bridging to different systems
 */
public interface TraceTaskScheduler {
    /**
     * Schedules a Runnable to run with the specified period
     *
     * @param command the task to run periodically
     * @param period  delay between first and successive runs of the task
     * @param unit    unit of time for period
     * @return a future that may be used to cancel the task
     */
    Future schedulePeriodicTask(Runnable command, long period, TimeUnit unit);

    /**
     * Default implementation uses a ScheduledExecutorService
     */
    class Default implements TraceTaskScheduler {
        private ScheduledExecutorService executor;

        /**
         * Creates a scheduler that uses exactly one background thread
         */
        public Default() {
            this(1);
        }

        /**
         * Creates a scheduler that uses the specified core pool size
         */
        public Default(int corePoolSize) {
            this(corePoolSize, corePoolSize > 0 ? corePoolSize : 1);
        }

        /**
         * Creates a scheduler that uses the specified core and maximum pool size
         */
        public Default(int corePoolSize, int maxPoolSize) {
            this(defaultExecutor(corePoolSize, maxPoolSize));
        }

        /**
         * Creates a scheduler that uses the specified executor
         */
        public Default(ScheduledExecutorService executor) {
            this.executor = executor;
        }

        /**
         * Returns a ScheduledThreadPoolExecutor with the specified core and maximum pool size,
         * which is configured to remove tasks on cancellation.
         */
        public static ScheduledExecutorService defaultExecutor(int corePoolSize, int maxPoolSize) {
            ScheduledThreadPoolExecutor executor = new ScheduledThreadPoolExecutor(corePoolSize);
            executor.setMaximumPoolSize(maxPoolSize);
            executor.setRemoveOnCancelPolicy(true);
            ThreadFactory threadFactory = new ThreadFactoryBuilder()
                    .setNameFormat("trace-bg-%d")
                    .build();
            executor.setThreadFactory(threadFactory);
            return executor;
        }

        @Override
        public Future schedulePeriodicTask(Runnable command, long period, TimeUnit unit) {
            return executor.scheduleWithFixedDelay(command, period, period, unit);
        }
    }
}
