package ru.yandex.direct.tracing.util;

import java.util.Comparator;
import java.util.Map;
import java.util.concurrent.ThreadLocalRandom;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.google.common.base.CharMatcher;
import com.google.common.base.Joiner;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceChild;
import ru.yandex.direct.tracing.real.RealTrace;

/**
 * Utility functions related to tracing
 */
public abstract class TraceUtil {
    private static final Logger logger = LoggerFactory.getLogger(TraceUtil.class);

    /**
     * HTTP header name used to relay tracing parameters.
     */
    public static final String X_YANDEX_TRACE = "X-Yandex-Trace";

    private static final int TRACE_SQL_COMMENT_MAX_PART_LENGTH = 64;
    private static final CharMatcher TRACE_SQL_COMMENT_INVALID_PART_CHARS =
            CharMatcher.inRange('0', '9')
                    .or(CharMatcher.inRange('a', 'z'))
                    .or(CharMatcher.inRange('A', 'Z'))
                    .or(CharMatcher.anyOf("_.-"))
                    .negate();

    private TraceUtil() {
    }

    /**
     * Returns a random number that may be used for trace identifiers
     */
    public static long randomId() {
        long value;
        do {
            // Lower part is random and may be 0
            // Higher part is 0 roughly every 25 days
            // Probability of both being 0 is low, but we still have to check for it
            value = ((System.currentTimeMillis() & 0x7fffffffL) << 31)
                    | (ThreadLocalRandom.current().nextInt() & 0xffffffffL);
        } while (value <= 0);
        return value;
    }

    /**
     * Returns number of seconds in specified nanoseconds.
     */
    public static double secondsFromNanoseconds(long nano) {
        return nano / 1_000_000_000.0;
    }

    /**
     * Creates trace object based on specified parameters.
     *
     * @param header  contents of the X-Yandex-Trace header, may be null
     * @param service traced service
     * @param method  traced method
     * @param tags    traced tags
     * @return trace object
     */
    public static RealTrace traceFromHeader(String header, String service, String method, String tags) {
        return traceFromHeader(header, service, method, tags, null);
    }

    /**
     * Creates trace object based on specified parameters using supplied {@link RealTrace.Builder} instance.
     *
     * @param header  contents of the X-Yandex-Trace header, may be null
     * @param service traced service
     * @param method  traced method
     * @param tags    traced tags
     * @param builder preconfigured RealTrace.Builder instance
     * @return trace object
     */
    public static RealTrace traceFromHeader(String header, String service, String method, String tags,
                                            RealTrace.Builder builder) {
        long traceId = 0;
        long parentId = 0;
        long spanId = 0;
        int ttl = 0;

        if (header != null) {
            String[] parts = header.split(",");
            try {
                if (parts.length > 0) {
                    traceId = Long.valueOf(parts[0]);
                }
                if (parts.length > 1) {
                    spanId = Long.valueOf(parts[1]);
                }
                if (parts.length > 2) {
                    parentId = Long.valueOf(parts[2]);
                }
                if (parts.length > 3) {
                    ttl = Integer.valueOf(parts[3]);
                }
            } catch (NumberFormatException e) {
                logger.warn("Failed to parse X-Yandex-Trace header: {}", header);
            }
        }

        if (spanId == 0) {
            // generate a valid trace id
            spanId = TraceUtil.randomId();
            if (parentId == 0) {
                // we didn't get a valid parent id
                parentId = traceId;
            }
        }

        if (traceId == 0) {
            if (parentId != 0) {
                // there's no valid trace id, however we can use parent id as the trace id
                traceId = parentId;
            } else {
                // there are no valid identifiers, we are the root trace
                traceId = spanId;
            }
        }

        if (ttl < 0) {
            // don't allow ttl less than zero
            ttl = 0;
        }

        RealTrace trace = (builder == null ? RealTrace.builder() : builder)
                .withIds(traceId, parentId, spanId)
                .withInfo(service, method, tags)
                .withTtl(ttl)
                .build();
        trace.setTtl(ttl);
        return trace;
    }

    /**
     * Creates trace object based on specified parameters.
     *
     * @param header  contents of the X-Yandex-Trace header, may be null
     * @param service traced service
     * @param method  traced method
     * @return trace object
     */
    public static RealTrace traceFromHeader(String header, String service, String method) {
        return traceFromHeader(header, service, method, null);
    }

    /**
     * Creates trace object based on specified parameters.
     *
     * @param header contents of the X-Yandex-Trace header, may be null
     * @return trace object
     */
    public static RealTrace traceFromHeader(String header) {
        return traceFromHeader(header, null, null);
    }

    /**
     * Transforms T to String, removes illegal symbols and truncates long strings
     */
    private static <T> String sanitize(T value) {
        String result = (value == null) ? "" : value.toString();
        result = TRACE_SQL_COMMENT_INVALID_PART_CHARS.replaceFrom(result, '_');
        result = result.length() < TRACE_SQL_COMMENT_MAX_PART_LENGTH
                ? result : result.substring(0, TRACE_SQL_COMMENT_MAX_PART_LENGTH);
        return result;
    }

    /**
     * MySQL/ClickHouse comment that should be inserted after SELECT/INSERT/UPDATE/DELETE keyword.
     * Logic must be equal to perl function <code>Yandex::DBTools::_add_trace_comment</code>
     *
     * @return Trace string into <code>{@literal /}{@literal *} {@literal *}{@literal /}</code>
     */
    public static String getTraceSqlComment() {
        Trace trace = Trace.current();
        TraceCommentVars traceCommentVars = TraceCommentVarsHolder.get();
        return "/* reqid:" +
                Stream.of(trace.getSpanId(), trace.getService(), trace.getMethod())
                        .map(TraceUtil::sanitize)
                        .collect(Collectors.joining(":"))
                + traceCommentVars.getMap().entrySet().stream()
                .sorted(Comparator.comparing(Map.Entry::getKey))
                .map(e -> ":" + sanitize(e.getKey()) + "=" + sanitize(e.getValue()))
                .collect(Collectors.joining(""))
                + " */";
    }

    /**
     * Получить по трейс-объекту значение для HTTP-заголовка {@link #X_YANDEX_TRACE}
     *
     * @param child дочерний трейс, параметрами которого будет заполнен заголовок
     * @return строка со значением для заголовка X-Yandex-Trace
     */
    public static String traceToHeader(TraceChild child) {
        return traceToHeader(child, child.getTtl());
    }

    /**
     * Получить по трейс-объекту значение для HTTP-заголовка {@link #X_YANDEX_TRACE}
     *
     * @param child дочерний трейс, параметрами которого будет заполнен заголовок
     * @param ttl
     * @return строка со значением для заголовка X-Yandex-Trace
     */
    public static String traceToHeader(TraceChild child, long ttl) {
        if (ttl < 0) {
            ttl = 0;
        }
        return Joiner.on(',').join(child.getTraceId(), child.getSpanId(), child.getParentId(), ttl);
    }
}
