package ru.yandex.direct.transfermanagerutils;

import java.time.Duration;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;

import com.google.common.base.Stopwatch;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpMethod;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.retry.RetryPolicy;
import org.springframework.retry.backoff.FixedBackOffPolicy;
import org.springframework.retry.policy.SimpleRetryPolicy;
import org.springframework.retry.support.RetryTemplate;
import org.springframework.web.client.RestTemplate;

import ru.yandex.direct.solomon.SolomonExternalSystemMonitorService;
import ru.yandex.direct.utils.ThreadUtils;
import ru.yandex.direct.utils.io.FileUtils;

import static ru.yandex.direct.solomon.SolomonResponseMonitorStatus.STATUS_2XX;
import static ru.yandex.direct.solomon.SolomonResponseMonitorStatus.STATUS_5XX;
import static ru.yandex.direct.transfermanagerutils.TransferManagerRetryListener.fillRetryContextHeader;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Клиент для взаимодействия с Transfer Manager.
 * <p>
 * Transfer Manager - инструмент для переливки таблиц между кластерами в Yt. TM имеет HTTP API, к которому обращается
 * данный клиент.
 */
public class TransferManager {
    public static final Duration TASK_PING_INTERVAL = Duration.ofSeconds(30);
    private static final Logger logger = LoggerFactory.getLogger(TransferManager.class);
    private static final ParameterizedTypeReference<Map<String, Object>> awaitResponseType =
            new ParameterizedTypeReference<>() {
            };

    private static final String EXTERNAL_SYSTEM = "transfer_manager";
    private static final String METHOD_START_TASK = "start_task";
    private static final String METHOD_PING_TASK = "ping_task";
    private static final SolomonExternalSystemMonitorService MONITOR_SERVICE = new SolomonExternalSystemMonitorService(
            EXTERNAL_SYSTEM,
            Set.of(METHOD_START_TASK, METHOD_PING_TASK)
    );

    private final TransferManagerConfig config;
    private final RestTemplate restTemplate;
    private final RetryTemplate retryTemplate;

    /**
     * См. {@link TransferManagerRetryListener}
     */
    private final RetryTemplate mutationRetryTemplate;

    public TransferManager(TransferManagerConfig config) {
        this.config = config;
        this.restTemplate = new RestTemplate();
        this.retryTemplate = new RetryTemplate();
        this.mutationRetryTemplate = new RetryTemplate();

        FixedBackOffPolicy backOffPolicy = new FixedBackOffPolicy();
        backOffPolicy.setBackOffPeriod(Duration.ofSeconds(30).toMillis());

        RetryPolicy retryPolicy = new SimpleRetryPolicy(config.maxRetries);

        retryTemplate.setBackOffPolicy(backOffPolicy);
        retryTemplate.setRetryPolicy(retryPolicy);

        mutationRetryTemplate.setBackOffPolicy(backOffPolicy);
        mutationRetryTemplate.setRetryPolicy(retryPolicy);
        mutationRetryTemplate.registerListener(new TransferManagerRetryListener());
    }

    public String queryTransferManager(TransferManagerJobConfig jobConfig) throws TransferManagerException {
        return queryTransferManager(Collections.singletonList(jobConfig)).get(0);
    }

    /**
     * Запуск новой таски в TM
     */
    public List<String> queryTransferManager(List<TransferManagerJobConfig> jobConfig) throws TransferManagerException {
        return mapList(jobConfig, conf -> {
            TransferManagerQuery query = new TransferManagerQuery(conf.getInputCluster(), conf.getInputTable(),
                    conf.getOutputCluster(), conf.getOutputTable());

            HttpHeaders headers = setHeaders();
            ResponseEntity<String> response;
            try {
                response = mutationRetryTemplate.execute(context -> {
                    fillRetryContextHeader(context, headers);
                    HttpEntity<TransferManagerQuery> entity = new HttpEntity<>(query, headers);
                    return restTemplate.postForEntity(config.url.toURI(), entity, String.class);
                });
                MONITOR_SERVICE.write(METHOD_START_TASK, STATUS_2XX);
            } catch (Exception e) {
                MONITOR_SERVICE.write(METHOD_START_TASK, STATUS_5XX);
                throw new TransferManagerException("Failed to start TM task", e);
            }
            String taskId = response.getBody();
            logger.info("Started TM task with id " + taskId);

            return taskId;
        });
    }

    /**
     * Ждет выполнения таски в ТМ, периодически пингуя ее через HTTP-ручку
     *
     * @param taskIds      ID таски в ТМ
     * @param maxAwaitTime Максимальное время выполнения таски после которого будет брошен
     *                     {@link TransferManagerException}
     */
    public Map<String, Boolean> await(List<String> taskIds, Duration maxAwaitTime) throws TransferManagerException {
        HttpHeaders headers = setHeaders();
        Map<String, Boolean> result = new HashMap<>();
        Stopwatch stopwatch = Stopwatch.createStarted();
        while (stopwatch.elapsed(TimeUnit.SECONDS) < maxAwaitTime.getSeconds()) {
            HttpEntity<String> entity = new HttpEntity<>("parameters", headers);
            Map<String, Map<String, Object>> responses = StreamEx.of(taskIds)
                    .filter(taskId -> !result.containsKey(taskId))
                    .toMap(taskId -> {
                        try {
                            String url = config.url.toString() + taskId + "/";
                            var response = retryTemplate.execute(context -> restTemplate
                                    .exchange(url, HttpMethod.GET, entity, awaitResponseType)
                                    .getBody());
                            MONITOR_SERVICE.write(METHOD_PING_TASK, STATUS_2XX);
                            return response;
                        } catch (Exception e) {
                            MONITOR_SERVICE.write(METHOD_PING_TASK, STATUS_5XX);
                            throw e;
                        }
                    });

            responses.forEach((taskId, resp) -> TransferManagerState.byName((String) resp.get("state"))
                    .ifPresent(state -> result.put(taskId, state == TransferManagerState.COMPLETED)));

            if (result.size() == taskIds.size()) {
                return result;
            }
            ThreadUtils.sleep(TASK_PING_INTERVAL);
        }
        throw new TransferManagerException("Some tasks failed to finish in a given time: " + maxAwaitTime.toString());
    }

    /**
     * Выставляет нужные для доступа к ТМ хедеры
     */
    private HttpHeaders setHeaders() {
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.set("Authorization", "OAuth " + FileUtils.slurp(config.tokenPath).replaceAll("\n", ""));
        // Здесь именно Accept-Type, а не Accept, если верить доке
        headers.set("Accept-Type", MediaType.APPLICATION_JSON_VALUE);
        return headers;
    }
}
