package ru.yandex.direct.turbolandings.client;

import java.time.Duration;
import java.util.Collection;
import java.util.List;

import one.util.streamex.StreamEx;

import ru.yandex.direct.asynchttp.FetcherSettings;
import ru.yandex.direct.asynchttp.ParallelFetcherFactory;
import ru.yandex.direct.asynchttp.Result;
import ru.yandex.direct.http.smart.core.Smart;
import ru.yandex.direct.turbolandings.client.model.DcTurboLanding;
import ru.yandex.direct.turbolandings.client.model.GetIdByUrlResponseItem;
import ru.yandex.direct.turbolandings.client.model.GetTurboLandingsResponse;

import static io.netty.handler.codec.http.HttpHeaderNames.AUTHORIZATION;
import static ru.yandex.direct.http.smart.error.ErrorUtils.checkResultForErrors;

/**
 * Клиент для получения данных турболендингов из конструктора
 */
public class TurboLandingsClient {
    private static final int DEFAULT_LIMIT = 150;
    private static final int DEFAULT_OFFSET = 0;

    private final Api api;
    private Duration requestTimeout = Duration.ofSeconds(3);

    public TurboLandingsClient(
            TurboLandingsClientConfiguration config,
            ParallelFetcherFactory parallelFetcherFactory
    ) {
        api = createApi(config, parallelFetcherFactory);
    }

    public TurboLandingsClient(
            TurboLandingsClientConfiguration config,
            ParallelFetcherFactory parallelFetcherFactory,
            Duration requestTimeout
    ) {
        this.requestTimeout = requestTimeout;
        api = createApi(config, parallelFetcherFactory);
    }

    private Api createApi(TurboLandingsClientConfiguration config, ParallelFetcherFactory parallelFetcherFactory) {
        return Smart.builder()
                .withParallelFetcherFactory(parallelFetcherFactory)
                .addHeaderConfigurator(headers -> headers.add(AUTHORIZATION, config.getToken()))
                .withProfileName("turbo_landings.client")
                .withBaseUrl(config.getUrl())
                .build()
                .create(Api.class);
    }

    public List<DcTurboLanding> getTurboLandings(Long clientId, Collection<Long> turbolandingIds) {
        String ids = turbolandingIds.isEmpty() ? null : StreamEx.of(turbolandingIds).joining(",");

        Result<GetTurboLandingsResponse> result = api.getTurboLandings(clientId, ids, DEFAULT_LIMIT, DEFAULT_OFFSET)
                .execute(new FetcherSettings().withRequestTimeout(requestTimeout));
        checkResultForErrors(result, TurboLandingsClientException::new);
        return result.getSuccess().getItems();
    }

    public List<GetIdByUrlResponseItem> getTurbolandingIdsByUrl(Collection<String> urls) {
        Result<List<GetIdByUrlResponseItem>> result = api.getTurbolandingIdsByUrl(urls).execute();
        checkResultForErrors(result, TurboLandingsClientException::new);
        return result.getSuccess();
    }
}
