package ru.yandex.direct.turboapps.client;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import NSuperApp.Turbo;
import com.google.protobuf.InvalidProtocolBufferException;
import com.google.protobuf.util.JsonFormat;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.asynchttp.ParallelFetcherFactory;
import ru.yandex.direct.asynchttp.Result;
import ru.yandex.direct.http.smart.converter.ResponseConverterFactory;
import ru.yandex.direct.http.smart.core.Smart;
import ru.yandex.direct.turboapps.client.model.TurboAppInfoRequest;
import ru.yandex.direct.turboapps.client.model.TurboAppInfoResponse;
import ru.yandex.direct.turboapps.client.model.TurboAppMetaContentResponse;
import ru.yandex.direct.tvm.TvmIntegration;
import ru.yandex.direct.tvm.TvmService;
import ru.yandex.direct.utils.JsonUtils;

import static com.google.common.collect.Lists.partition;
import static ru.yandex.direct.http.smart.error.ErrorUtils.checkResultForErrors;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;

public class TurboAppsClient {

    private static final Logger logger = LoggerFactory.getLogger(TurboAppsClient.class);

    private final Api api;
    private final JsonFormat.Printer jsonPrinter;

    private final int chunkSize;

    public TurboAppsClient(String url, int chunkSize, TvmIntegration tvmIntegration, TvmService tvmService,
                           ParallelFetcherFactory parallelFetcherFactory) {
        this.chunkSize = chunkSize;
        this.api = createApi(url, tvmIntegration, tvmService, parallelFetcherFactory);
        this.jsonPrinter = JsonFormat.printer().omittingInsignificantWhitespace();
    }

    private Api createApi(String url, TvmIntegration tvmIntegration, TvmService tvmService,
                          ParallelFetcherFactory parallelFetcherFactory) {
        return Smart.builder()
                .withParallelFetcherFactory(parallelFetcherFactory)
                .withResponseConverterFactory(ResponseConverterFactory.builder()
                        .addConverters(new TurboAppsResponseConverter())
                        .build())
                .withProfileName("turboapp_client")
                .useTvm(tvmIntegration, tvmService)
                .withBaseUrl(url)
                .build()
                .create(Api.class);
    }

    /**
     * @return bannerId -> turboAppInfo
     */
    public Map<Long, TurboAppInfoResponse> getTurboApps(List<TurboAppInfoRequest> requests) throws TurboAppsClientException {
        Map<Long, TurboAppInfoResponse> response = new HashMap<>();

        List<Turbo.TDirectURLsRequest> chunks = StreamEx.of(partition(requests, chunkSize))
                .map(chunk -> Turbo.TDirectURLsRequest.newBuilder()
                        .addAllData(StreamEx.of(chunk)
                                .map(item -> Turbo.TDirectURLsRequest.TBanner.newBuilder()
                                        .setBannerID(item.getBannerId())
                                        .setBannerURL(item.getBannerUrl())
                                        .build())
                                .toList())
                        .build())
                .toList();

        logger.info("TurboAppsClient request with {} urls: {}", requests.size(), JsonUtils.toJson(requests));

        var results = api.getTurboAppsInfo(chunks).execute(-1);

        for (Map.Entry<Long, Result<Turbo.TDirectURLsResponse>> chunkResponse : results.entrySet()) {
            var chunkResult = chunkResponse.getValue();
            checkResultForErrors(chunkResult, TurboAppsClientException::new);

            logger.info("Chunk: {}, response: {}", chunkResponse.getKey(), JsonUtils.toJson(chunkResult.getSuccess()));

            for (var banner : chunkResult.getSuccess().getDataList()) {
                if (banner.hasAppId()) {
                    var turboAppInfo = new TurboAppInfoResponse()
                            .withBannerId(banner.getBannerID())
                            .withBannerUrl(banner.getBannerURL())
                            .withAppId(banner.getAppId());

                    try {
                        turboAppInfo.setContent(jsonPrinter.print(banner.getContent()));
                        turboAppInfo.setMetaContent(jsonPrinter.print(banner.getMetaContent()));
                    } catch (InvalidProtocolBufferException e) {
                        throw new TurboAppsClientException("Failed to serialize banner content or meta-content", e);
                    }

                    response.put(banner.getBannerID(), turboAppInfo);
                }
            }
        }

        return response;
    }

    public TurboAppMetaContentResponse tryParseMetaContent(String metaContent) {
        try {
            return ifNotNull(metaContent, m -> JsonUtils.fromJson(m, TurboAppMetaContentResponse.class));
        } catch (IllegalArgumentException e) {
            logger.warn("Invalid json passed to tryParseMetaContent: {}", metaContent);
            return null;
        }
    }
}
